import { Condition } from "../Condition";
import {
  ConditionLeft,
  ConditionLeftType, ConditionRight,
  ConditionRightType,
  Device,
  DeviceChildren,
  ModuleDescriptorCondition,
  ModuleDescriptorSchema,
  ModuleRule,
  ModuleRuleEvent
} from './ConditionManagerTypes';
import { ConditionGroup, OperatorType } from "../ConditionGroup";
import { ConditionOperand } from "../ConditionOperands/ConditionOperand";
import { ConstOperand } from "../ConditionOperands/ConstOperand";
import { VariableOperand } from "../ConditionOperands/VariableOperand";
import { EventParameterOperand } from "../ConditionOperands/EventParameterOperand";

type ModuleDescriptorDevice = {
  children: DeviceChildren,
  extensions: DeviceChildren
}

export class ConditionManager {
  private conditionGroup:Record<string, ConditionGroup> = {};
  private moduleDescriptors: Record<string, ModuleDescriptorSchema> = {};


  constructor(private readonly instanceApiRef: any) {
    this.instanceApiRef = instanceApiRef;
  }

  loadConditions(): void {
    const moduleDescriptor: ModuleDescriptorSchema = this.instanceApiRef.getModuleData(this.instanceApiRef.instanceDir);

    for (const [, device] of Object.entries(moduleDescriptor.Devices)) {
      device.rules?.forEach((rule, index) => {

        //Create Any Condition Group
        const conditionGroupAny = this.fillConditionGroupWithConditions(
          OperatorType.Or,
          rule.conditions.any,
          rule,
          device
        );

        //Create Every ConditionGroup
        const conditionGroupEvery = this.fillConditionGroupWithConditions(
          OperatorType.And,
          rule.conditions.every,
          rule,
          device
        );

        //create new condition group and add any and every condition groups to it
        const conditionGroupFinal = new ConditionGroup(OperatorType.And);
        conditionGroupFinal.addCondition(conditionGroupAny);
        conditionGroupFinal.addCondition(conditionGroupEvery);

        this.conditionGroup["rule" + index] = conditionGroupFinal;

      });
    }
  }

  evaluateCondition(id: string, ...eventCallbackParameters: unknown[]): boolean {
    if (this.conditionGroup[id]) {
      return this.conditionGroup[id].evaluate(eventCallbackParameters);
    } else {
      throw Error(`Rule with ${id} ID does not have condition`);
    }
  }

  private fillConditionGroupWithConditions(
    conditionGroupOperatorType: OperatorType,
    conditionArray: ModuleDescriptorCondition[],
    rule: ModuleRule,
    device: Device
  ) : ConditionGroup {
    const conditionGroup = new ConditionGroup(conditionGroupOperatorType);
    conditionArray.forEach((condition) => {
      //create Left side
      const { leftOperand, leftOperandValueType } = this.createLeftOperand(condition.left, rule, device);

      //create Right side
      const rightOperand = this.createRightOperand(condition.right, leftOperandValueType);

      conditionGroup.addCondition(new Condition(leftOperand, condition.operator, rightOperand));
    });
    return conditionGroup;
  }

  private createLeftOperand(
    leftCondition: ConditionLeft,
    rule: ModuleRule,
    device: Device
  ): {
    leftOperand: ConditionOperand,
    leftOperandValueType: string
  } {
    switch (leftCondition.type) {
      case ConditionLeftType.EventCallbackParameter: {
        return {
          leftOperand: new EventParameterOperand(leftCondition.name, leftCondition.index as number),
          leftOperandValueType: this.getEventCallbackParameterType(
            this.getRemoteDevice(rule.event, device),
            rule.event.event,
            leftCondition.name
          )
        };
      }
      case ConditionLeftType.StatusVariable: {
        return {
          leftOperand: new VariableOperand(this.instanceApiRef, this.instanceApiRef.instanceId, leftCondition.name),
          leftOperandValueType: this.getVariableType(this.getRemoteDevice(leftCondition, device), leftCondition.name)
        };
      }
    }
  }

  private createRightOperand(rightCondition: ConditionRight, leftOperandValueType: string): ConditionOperand {
    //On the right side, only ConstOperand is supported for now
    switch (rightCondition.type) {
      case ConditionRightType.EventCallbackParameter:
        return new EventParameterOperand(rightCondition.name as string, rightCondition.index as number);
      case ConditionRightType.StatusVariable:
        return new VariableOperand(this.instanceApiRef, this.instanceApiRef.instanceId, rightCondition.name as string);
      case ConditionRightType.Constant:
        //leftOperandValueType 'label' | 'integer' | 'float' | 'string' | 'boolean' | 'json' | 'list' | 'file'
        return new ConstOperand(this.convertType(rightCondition.value, leftOperandValueType));
    }
  }

  private getRemoteDevice(leftConditionOrModuleRuleEvent: ConditionLeft | ModuleRuleEvent, device: Device): Device {
    // check that the variable is in other module or not
    if (leftConditionOrModuleRuleEvent.instanceId && leftConditionOrModuleRuleEvent.instanceId != '') {
      const remoteInstanceDir = this.instanceApiRef.instanceDir.replace(
        this.instanceApiRef.instanceId,
        leftConditionOrModuleRuleEvent.instanceId
      );
      if (!this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId]) {
        this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId] = this.instanceApiRef.getModuleData(
          remoteInstanceDir
        );
      }
      return this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId]
        .Devices[Object.keys(this.moduleDescriptors[leftConditionOrModuleRuleEvent.instanceId].Devices)[0]];
    } else {
      return device;
    }
  }

  private convertType(value: any, type: string): any {
    switch (type) {
      case 'integer':
      case 'float':
      case 'number': {
        const num = Number(value);
        if (num) {
          return num;
        } else {
          throw new Error(`Value type is ${type} but the value is not ${type}.`);
        }
      }
      case 'string':
        return value.toString();
      case 'boolean':
        if (value === true || value == 'true' || value === false || value == 'false') {
          return (value === true || value == 'true');
        } else {
          throw new Error(`Value type is boolean but the value is not boolean.`);
        }
      case 'json':
        if (typeof value === 'object') {
          JSON.stringify(value);
          return value;
        } else {
          return JSON.parse(value);
        }
      default:
        return value;
    }
  }

  private getVariableType(item: ModuleDescriptorDevice, variableName: string): string {
    const valueType = this.getVariableValueType(item.children, variableName);
    return valueType == '' ? this.getVariableValueType(item.extensions, variableName) : valueType;
  }

  private getVariableValueType(childrenOrExtensions: DeviceChildren, variableName: string): string {
    if (childrenOrExtensions) {
      for (const [key, value] of Object.entries(childrenOrExtensions)) {
        if (key == variableName && value.type == 'variable') {
          return value.valueType;
        }
      }
    }
    return '';
  }

  private getEventCallbackParameterType(
    item: ModuleDescriptorDevice,
    eventName: string,
    parameterName: string
  ): string {
    const paramType = this.getEventCallbackParameterValueType(item.children, eventName, parameterName);
    return paramType == '' ? this.getEventCallbackParameterValueType(
      item.extensions,
      eventName,
      parameterName
    ) : paramType;
  }

  private getEventCallbackParameterValueType(
    childrenOrExtensions: DeviceChildren,
    eventName: string,
    parameterName: string
  ): string {
    if (childrenOrExtensions) {
      for (const [eventKey, eventValue] of Object.entries(childrenOrExtensions)) {
        if (eventKey == eventName && eventValue.type == 'event') {
          for (const parameter of eventValue.callbackParameters) {
            if (parameter.name == parameterName) {
              return parameter.type;
            }
          }
        }
      }
    }
    return '';
  }
}
