"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const VariableAddedMessage_1 = require("../Communication/VariableAddedMessage");
const VariableChangedMessage_1 = require("../Communication/VariableChangedMessage");
const VariableGetRemoteResponse_1 = require("../Communication/VariableGetRemoteResponse");
const VariableMessageType_1 = require("../Communication/VariableMessageType");
const RemoteVariable_1 = require("../RemoteVariable");
describe('RemoteVariable', () => {
    afterEach(() => {
        jest.clearAllMocks();
    });
    const consoleWarnSpy = jest.spyOn(console, 'warn').mockImplementationOnce(() => { });
    test('setLastModifiedMs check if current last modified time is less then previous one', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        expect(remoteVariable.value).toBe(undefined);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const json = JSON.parse(JSON.stringify(jsonMessage));
        const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        const jsonMessage2 = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 60,
                previousValue: 10,
                lastModifiedMs: 12
            }
        };
        const json2 = JSON.parse(JSON.stringify(jsonMessage2));
        const convertedMessage2 = Object.setPrototypeOf(json2, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage2);
        expect(remoteVariable.value).toBe(60);
        expect(remoteVariable.previousValue).toBe(10);
        expect(remoteVariable.lastModifiedMs).toBe(12);
        expect(consoleWarnSpy).toBeCalledTimes(1);
        expect(consoleWarnSpy).toBeCalledWith(`The incoming timestamp (12 ms) is earlier then the current timestamp (123 ms) in the taurus.cica status variable!`);
    });
    test('Successfully create a RemoteVariable', () => {
        expect(() => {
            new RemoteVariable_1.RemoteVariable("cica", "ownerID");
        }).not.toThrow();
    });
    test('Cannot set remoteVariable', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "ownerID");
        expect(() => {
            remoteVariable.value = 10;
        }).not.toThrow();
        expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
        expect(consoleWarnSpy).toHaveBeenLastCalledWith("The value of LARA variable cica (owned by instance ownerID) can not be set . The value of a LARA variable can only be set by their owner.");
    });
    test('Check remoteVariable is valid', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "ownerID");
        expect(remoteVariable.isValid()).toBeFalsy();
    });
    test('get ownerID', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "ownerID");
        expect(remoteVariable.ownerId).toBe("ownerID");
    });
    test('onValid event called with onMessage(VariableChangedMessage)', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        const onValidHandler = jest.fn();
        remoteVariable.onValid(onValidHandler);
        expect(remoteVariable.value).toBe(undefined);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const json = JSON.parse(JSON.stringify(jsonMessage));
        const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        expect(onValidHandler).toHaveBeenCalledTimes(1);
        expect(remoteVariable.value).toBe(10);
        expect(remoteVariable.previousValue).toBe(20);
        expect(remoteVariable.lastModifiedMs).toBe(123);
    });
    test('onInvalid event called', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        const onInvalidHandler = jest.fn();
        remoteVariable.onInvalid(onInvalidHandler);
        expect(remoteVariable.value).toBe(undefined);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const jsonMessage2 = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: undefined,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        let json = JSON.parse(JSON.stringify(jsonMessage));
        let convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        json = JSON.parse(JSON.stringify(jsonMessage2));
        convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        expect(onInvalidHandler).toHaveBeenCalledTimes(1);
    });
    test('onChanged event called when variable changed, but not calledd when subscricption occurs.', () => {
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        const onChangedHandler = jest.fn();
        remoteVariable.onChanged(onChangedHandler);
        expect(remoteVariable.value).toBe(undefined);
        const jsonMessage = {
            messageId: 'msg1',
            instanceId: 'taurus',
            source: 'source',
            messageType: VariableMessageType_1.VariableMessageType.Changed,
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const json = JSON.parse(JSON.stringify(jsonMessage));
        const convertedMessage = Object.setPrototypeOf(json, VariableChangedMessage_1.VariableChangedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        expect(onChangedHandler).toHaveBeenCalledTimes(1);
        expect(remoteVariable.value).toBe(10);
        expect(remoteVariable.previousValue).toBe(20);
        expect(remoteVariable.lastModifiedMs).toBe(123);
    });
    test('onMessage event called with VariableGetRemoteResponse', () => {
        const setInternalStateSpy = jest.spyOn(RemoteVariable_1.RemoteVariable.prototype, 'setInternalState').mockImplementation(jest.fn());
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        const jsonMessage = {
            messageType: VariableMessageType_1.VariableMessageType.GetRemoteResponse,
            instanceId: 'taurus',
            source: 'subscriber1',
            messageId: 'msg1',
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const json = JSON.parse(JSON.stringify(jsonMessage));
        const convertedMessage = Object.setPrototypeOf(json, VariableGetRemoteResponse_1.VariableGetRemoteResponse.prototype);
        remoteVariable.onMessage(convertedMessage);
        expect(setInternalStateSpy).toHaveBeenCalledTimes(1);
    });
    test('onMessage event called with VariableAddedMessage', () => {
        const setInternalStateSpy = jest.spyOn(RemoteVariable_1.RemoteVariable.prototype, 'setInternalState').mockImplementation(jest.fn());
        const remoteVariable = new RemoteVariable_1.RemoteVariable("cica", "taurus");
        const jsonMessage = {
            messageType: VariableMessageType_1.VariableMessageType.Added,
            instanceId: 'taurus',
            source: 'subscriber1',
            messageId: 'msg1',
            messageData: {
                variableName: 'cica',
                ownerId: 'taurus',
                currentValue: 10,
                previousValue: 20,
                lastModifiedMs: 123
            }
        };
        const json = JSON.parse(JSON.stringify(jsonMessage));
        const convertedMessage = Object.setPrototypeOf(json, VariableAddedMessage_1.VariableAddedMessage.prototype);
        remoteVariable.onMessage(convertedMessage);
        expect(setInternalStateSpy).toHaveBeenCalledTimes(1);
    });
});
//# sourceMappingURL=RemoteVariable.test.js.map