import { EventEmitter } from "node:events";
import { Variable, VariableType, TypeSignature, StateChange } from "./Variable";
import { EventNames } from "./EventNames";

enum InternalEventNames {
  ChangedInternal = 'changedInternal',
}

export class LocalVariable implements Variable {
  private _currentValue: VariableType = undefined;
  private _previousValue: VariableType = undefined;

  private _lastModifiedMs: number = -1;

  private readonly eventEmitter: NodeJS.EventEmitter = new EventEmitter();

  constructor(public readonly name: string,
              public readonly typeSignature: TypeSignature,
              private readonly _ownerId: string,
              defaultValue: VariableType = undefined) {

    this.setInternalState(defaultValue);
  }

  get ownerId(): string {
    return this._ownerId;
  }

  get value(): VariableType {
    return this._currentValue;
  }

  set value(value: VariableType) {
    this.setInternalState(value);
  }

  get previousValue(): VariableType {
    return this._previousValue;
  }

  get lastModifiedMs(): number {
    return this._lastModifiedMs;
  }

  onValid(listener: () => void): void {
    this.eventEmitter.on(EventNames.Valid, listener);
  }

  onInvalid(listener: () => void): void {
    this.eventEmitter.on(EventNames.Invalid, listener);
  }

  onChanged(listener: (currentValue: VariableType, previousValue: VariableType) => void): void {
    this.eventEmitter.on(EventNames.Changed, listener);
  }

  onChangedInternal(listener: (variable: LocalVariable) => void): void {
    this.eventEmitter.on(InternalEventNames.ChangedInternal, listener);
  }

  private setInternalState(value: VariableType): void {
    // It is necessary to manage the variables of the factory modules
    if(value == null) {
      value = undefined;
    }

    this.checkAndWarnTypeSignature(value);

    if (this._currentValue != value) {
      let stateChange: StateChange = StateChange.None;

      if (this._currentValue === undefined && value !== undefined) {
        stateChange = StateChange.ToValid;
      } else if(this._currentValue !== undefined && value === undefined) {
        stateChange = StateChange.ToInvalid;
      }

      this._previousValue = this._currentValue;
      this._currentValue = value;
      this._lastModifiedMs = performance.now();

      if (stateChange == StateChange.ToValid) {
        this.eventEmitter.emit(EventNames.Valid);
      } else if (stateChange == StateChange.ToInvalid) {
        this.eventEmitter.emit(EventNames.Invalid);
      }

      this.eventEmitter.emit(EventNames.Changed, this._currentValue, this._previousValue);
      this.eventEmitter.emit(InternalEventNames.ChangedInternal, this);
    }
  }

  private checkAndWarnTypeSignature(value: VariableType): void {
    return;
    if (typeof value !== this.typeSignature && typeof value !== 'undefined') {
      console.warn(
        `The variable, ${this.ownerId}.${this.name}, is tried to be set with incompatible value! The provided value type is ${typeof value}, the expected type is ${this.typeSignature}`
      );
    }
  }
}