"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Condition_1 = require("./Condition");
const ConditionGroup_1 = require("./ConditionGroup");
const ConstOperand_1 = require("./ConditionOperands/ConstOperand");
const EventParameterOperand_1 = require("./ConditionOperands/EventParameterOperand");
const VariableOperand_1 = require("./ConditionOperands/VariableOperand");
// It is need for nodejs version 16, because the now property is readonly.
Object.defineProperty(performance, "now", {
    value: jest.fn(),
    configurable: true,
    writable: true
});
describe('Operator enum', () => {
    test('checks if a value is a valid key', () => {
        expect(Condition_1.Operator.isValidKey('==')).toBe(true);
        expect(Condition_1.Operator.isValidKey('!=')).toBe(true);
        expect(Condition_1.Operator.isValidKey('<')).toBe(true);
        expect(Condition_1.Operator.isValidKey('<=')).toBe(true);
        expect(Condition_1.Operator.isValidKey('>')).toBe(true);
        expect(Condition_1.Operator.isValidKey('>=')).toBe(true);
        expect(Condition_1.Operator.isValidKey('contains')).toBe(true);
        expect(Condition_1.Operator.isValidKey('regexp')).toBe(true);
        expect(Condition_1.Operator.isValidKey('invalid')).toBe(false);
    });
    test('checks if a value is a valid operator', () => {
        expect(Condition_1.Operator.isValidValue('equals')).toBe(true);
        expect(Condition_1.Operator.isValidValue('not_equals')).toBe(true);
        expect(Condition_1.Operator.isValidValue('less')).toBe(true);
        expect(Condition_1.Operator.isValidValue('less_or_equal')).toBe(true);
        expect(Condition_1.Operator.isValidValue('greater')).toBe(true);
        expect(Condition_1.Operator.isValidValue('greater_or_equal')).toBe(true);
        expect(Condition_1.Operator.isValidValue('contains')).toBe(true);
        expect(Condition_1.Operator.isValidValue('regexp')).toBe(true);
        expect(Condition_1.Operator.isValidValue('invalid')).toBe(false);
    });
});
describe('createCondition function', () => {
    test('should create a Condition with Constant operands', () => {
        const operandLhs = new ConstOperand_1.ConstOperand(5);
        const operator = Condition_1.Operator.Equals;
        const operandRhs = new ConstOperand_1.ConstOperand(5);
        const condition = (0, Condition_1.createCondition)(operandLhs, operator, operandRhs);
        expect(condition).toBeInstanceOf(Condition_1.Condition);
        expect(condition['operandLhs']).toBe(operandLhs);
        expect(condition['operator']).toBe(operator);
        expect(condition['operandRhs']).toBe(operandRhs);
    });
    test('should create a Condition with Variable operands', () => {
        const instanceApiMockL = {
            getInstanceById: jest.fn().mockImplementation(() => {
                return {
                    variables: {
                        test: {
                            value: 10,
                            lastModifiedMs: 1000,
                            onChanged: jest.fn()
                        }
                    }
                };
            })
        };
        const operandLhs = new VariableOperand_1.VariableOperand(instanceApiMockL, 'test', 'test');
        const instanceApiMockR = {
            getInstanceById: jest.fn().mockImplementation(() => {
                return {
                    variables: {
                        test: {
                            value: 5,
                            lastModifiedMs: 2000,
                            onChanged: jest.fn()
                        }
                    }
                };
            })
        };
        const operandRhs = new VariableOperand_1.VariableOperand(instanceApiMockR, 'test', 'test');
        const operator = Condition_1.Operator.Greater;
        const condition = (0, Condition_1.createCondition)(operandLhs, operator, operandRhs);
        expect(condition).toBeInstanceOf(Condition_1.Condition);
        expect(condition['operandLhs']).toBe(operandLhs);
        expect(condition['operator']).toBe(operator);
        expect(condition['operandRhs']).toBe(operandRhs);
    });
    test('should create a Condition with a numeric RHS value', () => {
        const operandLhs = new ConstOperand_1.ConstOperand(5);
        const operator = Condition_1.Operator.Less;
        const operandRhs = 10;
        const condition = (0, Condition_1.createCondition)(operandLhs, operator, operandRhs);
        expect(condition).toBeInstanceOf(Condition_1.Condition);
        expect(condition['operandLhs']).toBe(operandLhs);
        expect(condition['operator']).toBe(operator);
        expect(condition['operandRhs']).toBeInstanceOf(ConstOperand_1.ConstOperand);
        expect(condition['operandRhs']['value']).toBe(operandRhs);
    });
    test('should create a Condition with a string RHS value', () => {
        const operandLhs = new ConstOperand_1.ConstOperand(5);
        const operator = Condition_1.Operator.Less;
        const operandRhs = 'string';
        const condition = (0, Condition_1.createCondition)(operandLhs, operator, operandRhs);
        expect(condition).toBeInstanceOf(Condition_1.Condition);
        expect(condition['operandLhs']).toBe(operandLhs);
        expect(condition['operator']).toBe(operator);
        expect(condition['operandRhs']).toBeInstanceOf(ConstOperand_1.ConstOperand);
        expect(condition['operandRhs']['value']).toBe(operandRhs);
    });
    test('should create a Condition with a boolean RHS value', () => {
        const operandLhs = new ConstOperand_1.ConstOperand(5);
        const operator = Condition_1.Operator.Less;
        const operandRhs = true;
        const condition = (0, Condition_1.createCondition)(operandLhs, operator, operandRhs);
        expect(condition).toBeInstanceOf(Condition_1.Condition);
        expect(condition['operandLhs']).toBe(operandLhs);
        expect(condition['operator']).toBe(operator);
        expect(condition['operandRhs']).toBeInstanceOf(ConstOperand_1.ConstOperand);
        expect(condition['operandRhs']['value']).toBe(operandRhs);
    });
});
describe('Condition Class Test', () => {
    const eventParameters = [
        10,
        "foo",
        true,
        { "name": "John", "age": 30, "car": null }
    ];
    test('should throw error if the operator is invalid', () => {
        expect(() => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand(10), "====", new ConstOperand_1.ConstOperand(10)));
        }).toThrow("Invalid conditional operator of ====");
    });
    test('should throw error if rhs is not a ConstOperand', () => {
        const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
        expect(() => {
            conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Regexp, new EventParameterOperand_1.EventParameterOperand("arg", 0)));
        }).toThrow("Right Handside operand must be 'ConstOperand', if the operator is 'Regexp'");
    });
    test("should warn and set duration to 0 if lhs operand is not VariableOperand and duration > 0", () => {
        console.warn = jest.fn();
        const operandLhs = new ConstOperand_1.ConstOperand(5);
        const operandRhs = new ConstOperand_1.ConstOperand(10);
        const condition = new Condition_1.Condition(operandLhs, "equals", operandRhs, 100);
        expect(console.warn).toHaveBeenCalledWith("Warning: duration parameter can only be greater than zero if the left operand is variable!");
        expect(condition['duration']).toBe(0);
    });
    test('should return false if Eventparameter index is invalid', () => {
        const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
        expect(() => {
            conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("invalidName", 5), "==", new ConstOperand_1.ConstOperand(10)));
        }).not.toThrow();
        expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
    });
    describe('compareOperands', () => {
        const operatorEqualArr = [Condition_1.Operator.Equals, Condition_1.Operator.LessOrEqual, Condition_1.Operator.GreaterOrEqual];
        const operatorLessArr = [Condition_1.Operator.NotEquals, Condition_1.Operator.Less, Condition_1.Operator.LessOrEqual];
        const operatorGreaterArr = [Condition_1.Operator.Greater, Condition_1.Operator.GreaterOrEqual];
        let instanceApiMock /* eslint-enable */;
        let operandLhs;
        const duration = 500;
        beforeEach(() => {
            instanceApiMock = {
                getInstanceById: jest.fn().mockImplementation(() => {
                    return {
                        variables: {
                            test: {
                                value: 5,
                                lastModifiedMs: 1000,
                                onChanged: jest.fn()
                            }
                        }
                    };
                })
            };
            operandLhs = new VariableOperand_1.VariableOperand(instanceApiMock, 'test', 'test');
        });
        afterEach(() => {
            jest.clearAllMocks();
        });
        operatorEqualArr.forEach((operator) => {
            test(`should return true with ${operator} operator, when lhsValue is equals to rhsValue, and durationSinceLastChange is greater than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(5);
                const currentMs = 2000;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(true);
            });
            test(`should return false with ${operator} operator, when lhsValue is equals to rhsValue, and durationSinceLastChange is less than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(5);
                const currentMs = 1400;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(false);
            });
        });
        test(`should return false with Equals operator, when lhsValue is NOT equals to rhsValue, and durationSinceLastChange is greater than duration`, () => {
            const operandRhs = new ConstOperand_1.ConstOperand(10);
            const currentMs = 2000;
            jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
            const condition = new Condition_1.Condition(operandLhs, Condition_1.Operator.Equals, operandRhs, duration);
            const result = condition.evaluate(operandLhs.value, Condition_1.Operator.Equals, operandRhs.value);
            expect(result).toBe(false);
        });
        operatorLessArr.forEach((operator) => {
            test(`should return true with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is greater than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(10);
                const currentMs = 2000;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(true);
            });
            test(`should return false with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is less than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(10);
                const currentMs = 1400;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(false);
            });
            if (operator != Condition_1.Operator.NotEquals) {
                test(`should return false with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is greater than duration`, () => {
                    const operandRhs = new ConstOperand_1.ConstOperand(1);
                    const currentMs = 2000;
                    jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                    const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                    const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                    expect(result).toBe(false);
                });
            }
        });
        operatorGreaterArr.forEach((operator) => {
            test(`should return true with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is greater than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(1);
                const currentMs = 2000;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(true);
            });
            test(`should return false with ${operator} operator, when lhsValue is greater than rhsValue, and durationSinceLastChange is less than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(1);
                const currentMs = 1400;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(false);
            });
            test(`should return false with ${operator} operator, when lhsValue is less than rhsValue, and durationSinceLastChange is greater than duration`, () => {
                const operandRhs = new ConstOperand_1.ConstOperand(10);
                const currentMs = 2000;
                jest.spyOn(global.performance, "now").mockReturnValue(currentMs);
                const condition = new Condition_1.Condition(operandLhs, operator, operandRhs, duration);
                const result = condition.evaluate(operandLhs.value, operator, operandRhs.value);
                expect(result).toBe(false);
            });
        });
    });
    describe('evaluate', () => {
        test('should return true if number == number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), Condition_1.Operator.Equals, new ConstOperand_1.ConstOperand(10)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), Condition_1.Operator.Equals, new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if number == string', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "==", new ConstOperand_1.ConstOperand("10")));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
        });
        test('should return true if string == number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("10"), "==", new EventParameterOperand_1.EventParameterOperand("valNumber", 0)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
        });
        test('should return true if number < number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "<", new ConstOperand_1.ConstOperand(11)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "<", new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if number > number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), ">", new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), ">", new ConstOperand_1.ConstOperand(11)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if number != number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "!=", new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "!=", new ConstOperand_1.ConstOperand(10)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if number >= number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), ">=", new ConstOperand_1.ConstOperand(10)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), ">=", new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), ">=", new ConstOperand_1.ConstOperand(11)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if number <= number', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "<=", new ConstOperand_1.ConstOperand(10)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "<=", new ConstOperand_1.ConstOperand(11)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valNumber", 0), "<=", new ConstOperand_1.ConstOperand(9)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(false);
        });
        test('should return true if value == true', () => {
            const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
            expect(() => {
                conditionGroupAnd.addCondition(new Condition_1.Condition(new EventParameterOperand_1.EventParameterOperand("valBool", 2), "==", new ConstOperand_1.ConstOperand(true)));
            }).not.toThrow();
            expect(conditionGroupAnd.evaluate(...eventParameters)).toBe(true);
        });
        describe('String Contains', () => {
            test('should return true if string contains string', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Contains, new ConstOperand_1.ConstOperand("llo W")));
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(true);
            });
            test('should return false if string not contains string', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    const condition = new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Contains, new ConstOperand_1.ConstOperand("notContains"));
                    conditionGroupAnd.addCondition(condition);
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(false);
            });
            test('should return false if string not contains string', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    const condition = new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), 'contains', new ConstOperand_1.ConstOperand("notContains"));
                    conditionGroupAnd.addCondition(condition);
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(false);
            });
            test('should return true if string contains a number', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(async () => {
                    const condition = new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello 10  World"), Condition_1.Operator.Contains, new ConstOperand_1.ConstOperand(10));
                    conditionGroupAnd.addCondition(condition);
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(true);
            });
        });
        describe('Regexps', () => {
            test('should return false if pattern is invalid', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Regexp, new ConstOperand_1.ConstOperand('invalid')));
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(false);
            });
            test('Regexp w/o flags', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Regexp, new ConstOperand_1.ConstOperand("/^H.*d/")));
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(true);
            });
            test('Regexp w/o, false output', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Regexp, new ConstOperand_1.ConstOperand("/^H.*D/")));
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(false);
            });
            test('Regexp with flags', () => {
                const conditionGroupAnd = new ConditionGroup_1.ConditionGroup(ConditionGroup_1.OperatorType.And);
                expect(() => {
                    conditionGroupAnd.addCondition(new Condition_1.Condition(new ConstOperand_1.ConstOperand("Hello World"), Condition_1.Operator.Regexp, new ConstOperand_1.ConstOperand("/^h.*d/i")));
                }).not.toThrow();
                expect(conditionGroupAnd.evaluate([])).toBe(true);
            });
        });
    });
});
//# sourceMappingURL=Condition.test.js.map