"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ConstOperand_1 = require("../ConditionOperands/ConstOperand");
const EventParameterOperand_1 = require("../ConditionOperands/EventParameterOperand");
const VariableOperand_1 = require("../ConditionOperands/VariableOperand");
const ConditionManager_1 = require("./ConditionManager");
const Condition_1 = require("../Condition");
const ConditionManagerTypes_1 = require("./ConditionManagerTypes");
const ConditionGroup_1 = require("../ConditionGroup");
describe('ConditionManager', () => {
    const createDefaultModuleDescriptor = () => ({
        Schemas: {},
        Devices: {
            test: {
                name: '',
                type: 'group',
                children: {},
                rules: [],
                extensions: {}
            }
        }
    });
    test('Successfully create a ConditionManager instance', () => {
        expect(() => {
            new ConditionManager_1.ConditionManager({});
        }).not.toThrow();
    });
    describe('getVariableType', () => {
        test('successfully returns variable type if variable is in children', () => {
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testVariable: {
                    type: 'variable',
                    valueType: 'string',
                    defaultValue: '',
                    description: '',
                    orderId: 0,
                    factory: true
                }
            };
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const valueType = conditionManager['getVariableType'](moduleDescriptor.Devices.test, 'testVariable');
            expect(valueType).toBe('string');
        });
        test('successfully returns variable type if variable is in extensions', () => {
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.extensions = {
                testVariable: {
                    type: 'variable',
                    valueType: 'string',
                    defaultValue: '',
                    description: '',
                    orderId: 0,
                    factory: true
                }
            };
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const valueType = conditionManager['getVariableType'](moduleDescriptor.Devices.test, 'testVariable');
            expect(valueType).toBe('string');
        });
    });
    describe('convertType', () => {
        let conditionManager;
        beforeEach(() => {
            conditionManager = new ConditionManager_1.ConditionManager({});
        });
        test('should convert integer type correctly', () => {
            const value = '42';
            const type = 'integer';
            const result = conditionManager['convertType'](value, type);
            expect(result).toBe(42);
        });
        test('should convert float type correctly', () => {
            const value = '3.14';
            const type = 'float';
            const result = conditionManager['convertType'](value, type);
            expect(result).toBe(3.14);
        });
        test('should throw an error for non-numeric value in number type', () => {
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const value = 'invalid';
            const type = 'number';
            expect(() => {
                conditionManager['convertType'](value, type);
            }).toThrow('Value type is number but the value is not number.');
        });
        test('successfully converts string into boolean', () => {
            const result = conditionManager['convertType']('true', 'boolean');
            expect(typeof result).toBe('boolean');
            expect(result).toBe(true);
        });
        test('should throw an error for invalid boolean value', () => {
            const value = 'invalid';
            const type = 'boolean';
            expect(() => {
                conditionManager['convertType'](value, type);
            }).toThrow('Value type is boolean but the value is not boolean.');
        });
        test('successfully converts json object which is in a string to js object', () => {
            const jsonObjectString = '{ "test": "asdf" }';
            const result = conditionManager['convertType'](jsonObjectString, 'json');
            expect(typeof result).toBe('object');
            expect(result).toEqual({ test: 'asdf' });
        });
        test('successfully returns json object which is a js object', () => {
            const jsonObject = { test: 'asdf' };
            const result = conditionManager['convertType'](jsonObject, 'json');
            expect(typeof result).toBe('object');
            expect(result).toEqual(jsonObject);
        });
        test('throws an error if an invalid json object string is passed', () => {
            const jsonObjectString = '{ "test": "asdf }';
            expect(() => {
                conditionManager['convertType'](jsonObjectString, 'json');
            }).toThrow();
        });
    });
    describe('getEventCallbackParameterType', () => {
        test('successfully returns event callback paramter type if found in children', () => {
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testEvent: {
                    name: '',
                    type: 'event',
                    callbackParameters: [{
                            title: '',
                            orderId: 0,
                            name: 'testParam',
                            type: 'string',
                            description: ''
                        }],
                    parameterValues: [],
                    description: '',
                    orderId: 0,
                    subscriptionMethodCode: '',
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    factory: true
                }
            };
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const parameterType = conditionManager['getEventCallbackParameterType'](moduleDescriptor.Devices.test, 'testEvent', 'testParam');
            expect(parameterType).toBe('string');
        });
        test('successfully returns event callback paramter type if found in extensions', () => {
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.extensions = {
                testEvent: {
                    name: '',
                    type: 'event',
                    callbackParameters: [{
                            title: '',
                            orderId: 0,
                            name: 'testParam',
                            type: 'string',
                            description: ''
                        }],
                    parameterValues: [],
                    description: '',
                    orderId: 0,
                    subscriptionMethodCode: '',
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    factory: true
                }
            };
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const parameterType = conditionManager['getEventCallbackParameterType'](moduleDescriptor.Devices.test, 'testEvent', 'testParam');
            expect(parameterType).toBe('string');
        });
        test('returns empty string if event not found in moduleDescriptor', () => {
            const moduleDescriptor = createDefaultModuleDescriptor();
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const parameterType = conditionManager['getEventCallbackParameterType'](moduleDescriptor.Devices.test, 'testEvent', 'testParam');
            expect(parameterType).toBe('');
        });
    });
    describe('createRightOperand', () => {
        // TODO: test for VariableOperands
        let conditionManager;
        beforeEach(() => {
            conditionManager = new ConditionManager_1.ConditionManager({});
        });
        test('should create EventParameterOperand for ConditionRightType.EventCallbackParameter', () => {
            const rightCondition = {
                type: ConditionManagerTypes_1.ConditionRightType.EventCallbackParameter,
                name: 'testCondition',
                value: 'test'
            };
            const leftOperandValueType = 'string';
            const rightOperand = conditionManager['createRightOperand'](rightCondition, leftOperandValueType);
            expect(rightOperand).toBeInstanceOf(EventParameterOperand_1.EventParameterOperand);
        });
        test('returns a ConstOperand if conditionType is constant', () => {
            const rightCondition = {
                type: ConditionManagerTypes_1.ConditionRightType.Constant,
                name: 'testCondition',
                value: 'test'
            };
            const rightOperand = conditionManager['createRightOperand'](rightCondition, 'string');
            expect(rightOperand).toBeInstanceOf(ConstOperand_1.ConstOperand);
        });
    });
    describe('createLeftOperand', () => {
        test('returns left EventCallbackParameter operand if event is on its own instance', () => {
            const leftCondition = {
                type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                instanceId: '',
                name: 'testParam'
            };
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testEvent: {
                    name: '',
                    type: 'event',
                    callbackParameters: [{
                            title: '',
                            orderId: 0,
                            name: 'testParam',
                            type: 'string',
                            description: ''
                        }],
                    parameterValues: [],
                    description: '',
                    orderId: 0,
                    subscriptionMethodCode: '',
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    factory: true
                }
            };
            moduleDescriptor.Devices.test.rules = [{
                    name: 'testRule',
                    event: {
                        instanceId: '',
                        event: 'testEvent'
                    },
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    conditions: {
                        any: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                    instanceId: '',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test2',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }],
                        every: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                    instanceId: '',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test4',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }]
                    }
                }];
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const { leftOperand, leftOperandValueType } = conditionManager['createLeftOperand'](leftCondition, moduleDescriptor.Devices.test.rules[0], moduleDescriptor.Devices.test);
            expect(leftOperand instanceof EventParameterOperand_1.EventParameterOperand).toBe(true);
            expect(leftOperandValueType).toBe('string');
        });
        test('returns left EventCallbackParameter operand if event is on another instance', () => {
            const leftCondition = {
                type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                instanceId: '',
                name: 'testParam'
            };
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testEvent: {
                    name: '',
                    type: 'event',
                    callbackParameters: [{
                            title: '',
                            orderId: 0,
                            name: 'testParam',
                            type: 'string',
                            description: ''
                        }],
                    parameterValues: [],
                    description: '',
                    orderId: 0,
                    subscriptionMethodCode: '',
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    factory: true
                }
            };
            moduleDescriptor.Devices.test.rules = [{
                    name: 'testRule',
                    event: {
                        instanceId: 'test',
                        event: 'testEvent'
                    },
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    conditions: {
                        any: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                    instanceId: 'test',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test2',
                                    instanceId: 'test',
                                    value: 'asdf'
                                }
                            }],
                        every: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                    instanceId: 'test',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test4',
                                    instanceId: 'test',
                                    value: 'asdf'
                                }
                            }]
                    }
                }];
            const mockInstanceApi = {
                getModuleData: jest.fn().mockReturnValueOnce(moduleDescriptor),
                instanceDir: '/path/test2',
                instanceId: 'test2'
            };
            const conditionManager = new ConditionManager_1.ConditionManager(mockInstanceApi);
            const { leftOperand, leftOperandValueType } = conditionManager['createLeftOperand'](leftCondition, moduleDescriptor.Devices.test.rules[0], moduleDescriptor.Devices.test);
            expect(leftOperand instanceof EventParameterOperand_1.EventParameterOperand).toBe(true);
            expect(leftOperandValueType).toBe('string');
        });
        test('returns left VariableOperand operand if variable is on the same instance', () => {
            const leftCondition = {
                type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                instanceId: '',
                name: 'testVariable'
            };
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testVariable: {
                    type: "variable",
                    valueType: "string",
                    defaultValue: "Hello",
                    description: "",
                    orderId: 0,
                    factory: true
                }
            };
            moduleDescriptor.Devices.test.rules = [{
                    name: 'testRule',
                    event: {
                        instanceId: '',
                        event: 'testEvent'
                    },
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    conditions: {
                        any: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                                    instanceId: '',
                                    name: 'testVariable'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test2',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }],
                        every: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                                    instanceId: '',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test4',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }]
                    }
                }];
            const mockInstanceApi = { getInstanceById: jest.fn().mockReturnValue({ variables: { testVariable: { onChanged: jest.fn() } } }) };
            const conditionManager = new ConditionManager_1.ConditionManager(mockInstanceApi);
            const { leftOperand, leftOperandValueType } = conditionManager['createLeftOperand'](leftCondition, moduleDescriptor.Devices.test.rules[0], moduleDescriptor.Devices.test);
            expect(leftOperand).toBeInstanceOf(VariableOperand_1.VariableOperand);
            expect(leftOperandValueType).toBe('string');
        });
        test('returns left VariableOperand operand if variable is on a different instance', () => {
            const leftCondition = {
                type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                instanceId: '',
                name: 'testVariable'
            };
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testVariable: {
                    type: "variable",
                    valueType: "string",
                    defaultValue: "Hello",
                    description: "",
                    orderId: 0,
                    factory: true
                }
            };
            moduleDescriptor.Devices.test.rules = [{
                    name: 'testRule',
                    event: {
                        instanceId: '',
                        event: 'testEvent'
                    },
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    conditions: {
                        any: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                                    instanceId: '',
                                    name: 'testVariable'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test2',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }],
                        every: [{
                                left: {
                                    type: ConditionManagerTypes_1.ConditionLeftType.StatusVariable,
                                    instanceId: '',
                                    name: 'testParam'
                                },
                                operator: Condition_1.Operator.Equals,
                                right: {
                                    type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                    name: 'test4',
                                    instanceId: '',
                                    value: 'asdf'
                                }
                            }]
                    }
                }];
            const mockInstanceApi = {
                getModuleData: jest.fn().mockReturnValueOnce(moduleDescriptor),
                instanceDir: '/path/test2',
                instanceId: 'test2',
                getInstanceById: jest.fn().mockReturnValue({ variables: { testVariable: { onChanged: jest.fn() } } })
            };
            const conditionManager = new ConditionManager_1.ConditionManager(mockInstanceApi);
            const { leftOperand, leftOperandValueType } = conditionManager['createLeftOperand'](leftCondition, moduleDescriptor.Devices.test.rules[0], moduleDescriptor.Devices.test);
            expect(leftOperand instanceof VariableOperand_1.VariableOperand).toBe(true);
            expect(leftOperandValueType).toBe('string');
        });
    });
    describe('fillConditionGroupWithConditions', () => {
        test('successfully returns ConditionGroup with Conditions', () => {
            const conditionArray = [{
                    left: {
                        type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                        instanceId: '',
                        name: 'testParam1'
                    },
                    operator: Condition_1.Operator.Equals,
                    right: {
                        type: ConditionManagerTypes_1.ConditionRightType.Constant,
                        name: 'test1',
                        instanceId: '',
                        value: 'value1'
                    }
                },
                {
                    left: {
                        type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                        instanceId: '',
                        name: 'testParam1'
                    },
                    operator: Condition_1.Operator.Equals,
                    right: {
                        type: ConditionManagerTypes_1.ConditionRightType.Constant,
                        name: 'test2',
                        instanceId: '',
                        value: 'value2'
                    }
                }];
            const moduleDescriptor = createDefaultModuleDescriptor();
            moduleDescriptor.Devices.test.children = {
                testEvent: {
                    name: '',
                    type: 'event',
                    callbackParameters: [{
                            title: '',
                            orderId: 0,
                            name: 'testParam',
                            type: 'string',
                            description: ''
                        }],
                    parameterValues: [],
                    description: '',
                    orderId: 0,
                    subscriptionMethodCode: '',
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    factory: true
                }
            };
            moduleDescriptor.Devices.test.rules = [{
                    name: 'testRule',
                    event: {
                        instanceId: '',
                        event: 'testEvent'
                    },
                    actions: {
                        code: '',
                        steps: [],
                        type: 'wizard'
                    },
                    conditions: {
                        any: conditionArray,
                        every: []
                    }
                }];
            const conditionManager = new ConditionManager_1.ConditionManager({});
            const conditionGroup = conditionManager['fillConditionGroupWithConditions'](ConditionGroup_1.OperatorType.And, conditionArray, moduleDescriptor.Devices.test.rules[0], moduleDescriptor.Devices.test);
            expect(conditionGroup).toBeDefined();
        });
    });
    describe('loadAndEvaluateCondition', () => {
        const moduleDescriptor = createDefaultModuleDescriptor();
        moduleDescriptor.Devices.test.children = {
            testEvent: {
                name: '',
                type: 'event',
                callbackParameters: [{
                        title: '',
                        orderId: 0,
                        name: 'testParam',
                        type: 'string',
                        description: ''
                    }],
                parameterValues: [],
                description: '',
                orderId: 0,
                subscriptionMethodCode: '',
                actions: {
                    code: '',
                    steps: [],
                    type: 'wizard'
                },
                factory: true
            }
        };
        moduleDescriptor.Devices.test.rules = [{
                name: 'testRule',
                event: {
                    instanceId: '',
                    event: 'testEvent'
                },
                actions: {
                    code: '',
                    steps: [],
                    type: 'wizard'
                },
                conditions: {
                    any: [{
                            left: {
                                type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                instanceId: '',
                                name: 'testParam',
                                index: 0
                            },
                            operator: Condition_1.Operator.Equals,
                            right: {
                                type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                name: 'test2',
                                instanceId: '',
                                value: 'asdf'
                            }
                        }],
                    every: [{
                            left: {
                                type: ConditionManagerTypes_1.ConditionLeftType.EventCallbackParameter,
                                instanceId: '',
                                name: 'testParam',
                                index: 0
                            },
                            operator: Condition_1.Operator.Equals,
                            right: {
                                type: ConditionManagerTypes_1.ConditionRightType.Constant,
                                name: 'test4',
                                instanceId: '',
                                value: 'asdf'
                            }
                        }]
                }
            }];
        const mockInstanceApi = {
            getModuleData: jest.fn().mockReturnValue(moduleDescriptor),
            instanceDir: '/path/test',
            instanceId: 'test'
        };
        const conditionManager = new ConditionManager_1.ConditionManager(mockInstanceApi);
        test('successfully loads and evaluates condition', () => {
            conditionManager.loadConditions();
            expect(conditionManager.evaluateCondition('rule0', 'asdf')).toBe(true);
        });
        test('should throw an error if the condition group with the given id does not exist', () => {
            expect(() => {
                conditionManager.evaluateCondition('nonexistent-rule', {});
            }).toThrowError('Rule with nonexistent-rule ID does not have condition');
        });
    });
});
//# sourceMappingURL=ConditionManager.test.js.map