import { ConditionGroup, OperatorType } from './ConditionGroup';
import { Condition } from './Condition';
import { ConstOperand } from './ConditionOperands/ConstOperand';

describe('ConditionGroup', () => {

  test('successfully create ConditionGroup with and type', () => {
    expect(() => {
      new ConditionGroup(OperatorType.And);
    }).not.toThrow();
  });

  test('successfully create ConditionGroup with or type', () => {
    expect(() => {
      new ConditionGroup(OperatorType.Or);
    }).not.toThrow();
  });

  test('successfully add Condition to the ConditionGroup', () => {
    const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
    const conditionGroup = new ConditionGroup(OperatorType.And);

    expect(() => {
      conditionGroup.addCondition(condition);
    }).not.toThrow();
  });

  test('successfully add ConditionGroups to the ConditionGroup', () => {
    const conditionGroup1 = new ConditionGroup(OperatorType.And);
    const conditionGroup2 = new ConditionGroup(OperatorType.And);

    expect(() => {
      conditionGroup1.addCondition(conditionGroup2);
    }).not.toThrow();
  });

  test('successfully add a ConditionGroup and a Condition to the ConditionGroup', () => {
    const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
    const conditionGroup1 = new ConditionGroup(OperatorType.And);
    const conditionGroup2 = new ConditionGroup(OperatorType.And);

    expect(() => {
      conditionGroup1.addCondition(condition).addCondition(conditionGroup2);
    }).not.toThrow();
  });

  test('evaluate successfully calls evaluate with parameters on added Conditions and ConditionGroups', () => {
    const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
    const conditionGroup1 = new ConditionGroup(OperatorType.And);
    const conditionGroup2 = new ConditionGroup(OperatorType.And);
    conditionGroup1.addCondition(condition).addCondition(conditionGroup2);
    const eventParameters = ['test'];
    const conditionEvaluateSpy = jest.spyOn(condition, 'evaluate');
    const conditionGroupEvaluateSpy = jest.spyOn(conditionGroup1, 'evaluate');

    conditionGroup1.evaluate(eventParameters);

    expect(conditionEvaluateSpy).toHaveBeenCalledWith(eventParameters);
    expect(conditionGroupEvaluateSpy).toHaveBeenCalledWith(eventParameters);
  });

  describe('and type', () => {
    test('return true if an empty conditionGroup is evaluated', () => {
      const conditionGroup = new ConditionGroup(OperatorType.And);

      expect(conditionGroup.evaluate([])).toBe(true);
    });

    test('return true when calling evaluate with 2 true conditions', () => {
      const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
      const conditionGroup = new ConditionGroup(OperatorType.And);
      conditionGroup.addCondition(condition).addCondition(condition);

      expect(conditionGroup.evaluate([])).toBe(true);
    });

    test('return false when calling evaluate with 1 true 1 false condition', () => {
      const trueCondition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
      const falseCondition = new Condition(new ConstOperand(10), "==", new ConstOperand(1));
      const conditionGroup = new ConditionGroup(OperatorType.And);
      conditionGroup.addCondition(trueCondition).addCondition(falseCondition);

      expect(conditionGroup.evaluate([])).toBe(false);
    });

    test('return false when calling evaluate with 2 false conditions', () => {
      const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(1));
      const conditionGroup = new ConditionGroup(OperatorType.And);
      conditionGroup.addCondition(condition).addCondition(condition);

      expect(conditionGroup.evaluate([])).toBe(false);
    });
  });

  describe('or type', () => {
    test('return true if an empty conditionGroup is evaluated', () => {
      const conditionGroup = new ConditionGroup(OperatorType.Or);

      expect(conditionGroup.evaluate([])).toBe(true);
    });

    test('return true when calling evaluate with 2 true conditions', () => {
      const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
      const conditionGroup = new ConditionGroup(OperatorType.Or);
      conditionGroup.addCondition(condition).addCondition(condition);

      expect(conditionGroup.evaluate([])).toBe(true);
    });

    test('return true when calling evaluate with 1 true 1 false condition', () => {
      const trueCondition = new Condition(new ConstOperand(10), "==", new ConstOperand(10));
      const falseCondition = new Condition(new ConstOperand(10), "==", new ConstOperand(1));
      const conditionGroup = new ConditionGroup(OperatorType.Or);
      conditionGroup.addCondition(trueCondition).addCondition(falseCondition);

      expect(conditionGroup.evaluate([])).toBe(true);
    });

    test('return false when calling evaluate with 2 false conditions', () => {
      const condition = new Condition(new ConstOperand(10), "==", new ConstOperand(1));
      const conditionGroup = new ConditionGroup(OperatorType.Or);
      conditionGroup.addCondition(condition).addCondition(condition);

      expect(conditionGroup.evaluate([])).toBe(false);
    });
  });

  test("should catch and log errors and return false", () => {
    const conditionGroup = new ConditionGroup(OperatorType.And);
    class ErrorCondition {
      evaluate(): boolean {
        throw new Error("test error");
      }
    }
    conditionGroup.addCondition(new ErrorCondition());
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    const consoleSpy = jest.spyOn(console, "error").mockImplementation(() => {});
    const result = conditionGroup.evaluate([]);

    expect(result).toBe(false);
    expect(consoleSpy).toHaveBeenCalledWith("test error");
    consoleSpy.mockRestore();
  });

});
