import { ValueType } from "./ModuleDescriptorObject";

export class ActionsDescriptor {
  steps: StepDescriptor[];

  constructor(
    readonly code: string = '',
    readonly type: ActionType = ActionType.Code,
    steps: StepDescriptor[] = []
  ) {
    this.steps = steps;
    if (typeof this.steps != 'object') this.steps = [];
    this.migrateSteps();
  }

  migrateConstParameter(value: ValueType): ConstantParameter {
    return { parameterType: StepParameterType.Constant, value };
  }

  migrateParameter(value: (StepParameter[] | StepParameter)): StepParameter[] | StepParameter {
    return Array.isArray(value) ?
      value.map((v: any) => {
        if ('type' in v) {
          return this.migrateConstParameter(v.value);
        }
        return this.migrateConstParameter(v);
      }) :
      this.migrateConstParameter(value);
  }

  migrateMethodParameters(s: StepDescriptor, property: string): StepDescriptor {
    const castedStep = s.step as { [key: string]: any };

    if (castedStep[property].length) {
      if (castedStep[property][0]?.parameterType) {
        return s;
      }
    } else if (castedStep[property].parameterType) {
      return s;
    }

    return {
      ...s,
      step: {
        ...s.step,
        [property]: this.migrateParameter(castedStep[property])
      }
    };
  }

  migrateSteps(): void {
    this.steps = this.steps.map((s) => {
      switch (true) {
        case 'methodParameters' in s.step:
          return this.migrateMethodParameters(s, 'methodParameters');
        case 'eventParameters' in s.step:
          return this.migrateMethodParameters(s, 'eventParameters');
        case 'waitTimeMs' in s.step:
          return this.migrateMethodParameters(s, 'waitTimeMs');
        case 'message' in s.step:
          return this.migrateMethodParameters(s, 'message');
        case 'variableValue' in s.step:
          return this.migrateMethodParameters(s, 'variableValue');
        case 'text' in s.step:
          return this.migrateMethodParameters(s, 'text');
        case 'innerHtml' in s.step:
          return this.migrateMethodParameters(s, 'innerHtml');
        case 'cssPropertyValue' in s.step:
          return this.migrateMethodParameters(s, 'cssPropertyValue');
        case 'display' in s.step:
          return s;
        case 'value' in s.step:
          return this.migrateMethodParameters(s, 'value');
        case 'cssClassName' in s.step:
          return this.migrateMethodParameters(s, 'cssClassName');
        case 'htmlElementValue' in s.step:
          return this.migrateMethodParameters(s, 'htmlElementValue');
        default:
          throw new Error('Unknown step read');
      }
    });
  }
}

export enum ActionType {
  Wizard = 'wizard',
  Code = 'code'
}

export type StepDescriptor = {
  name: string;
  description: string;
  type: StepType;
  step:
  | MethodCallStep
  | SetVariableStep
  | EventEmitStep
  | TimeoutStep
  | LogStep
  | HtmlTextStep
  | HtmlInnerStep
  | HtmlStyleStep
  | HtmlHideStep
  | UpdateStatusStep
  | HtmlClassStep
  | HtmlElementValueStep;
};

export enum StepType {
  MethodCall = 'MethodCall',
  EventEmit = 'EventEmit',
  Timeout = 'Timeout',
  Log = 'Log',
  SetVariable = 'SetVariable'
}

export type SetVariableStep = {
  instanceId: string;
  variableName: string;
  variableValue: StepParameter;
  variableType: string
}

export type MethodCallStep = {
  instanceId: string;
  methodName: string;
  methodParameters: StepParameter[];
};

export type EventEmitStep = {
  eventName: string;
  eventParameters: StepParameter[];
};

export type TimeoutStep = {
  waitTimeMs: StepParameter;
};

export type LogStep = {
  message: StepParameter;
};

export type HtmlTextStep = {
  elementId: string;
  text: StepParameter;
}

export type HtmlInnerStep = {
  elementId: string;
  innerHtml: StepParameter;
}

export type HtmlStyleStep = {
  elementId: string;
  cssPropertyName: string;
  cssPropertyValue: StepParameter;
}

export type HtmlHideStep = {
  elementId: string;
  display: string;
}

export enum UpdateStatusStepTheme {
  Default = "Default",
  Success = "Success",
  Warning = "Warning",
  Error = "Error",
}

export type UpdateStatusStep = {
  label: string;
  value: StepParameter;
  themeColor: UpdateStatusStepTheme;
}

export type HtmlClassStep = {
  elementId: string;
  cssClassName: StepParameter;
}

export type HtmlElementValueStep = {
  elementId: string;
  htmlElementValue: StepParameter;
}

export enum StepParameterType {
  EventParameter = 'eventParameter',
  Constant = 'constant',
  InstanceParameter = 'instanceParameter'
}

export type StepParameter = EventParameter | ConstantParameter | InstanceParameter;

export type EventParameter = {
  parameterType: StepParameterType.EventParameter;
  paramName: string;
};

export type ConstantParameter = {
  parameterType: StepParameterType.Constant;
  value: ValueType;
};

export type InstanceParameter = {
  parameterType: StepParameterType.InstanceParameter;
  paramName: string;
}
