import { ConditionGroup, OperatorType, VariableOperand, createCondition } from "@lightware/condition-api";
import { RemoteVariable, TypeSignature } from "@lightware/variable-api";
import { VariableChangedMessage } from "@lightware/variable-api/dist/src/Communication/VariableChangedMessage";
import { VariableStateMessageData } from "@lightware/variable-api/dist/src/Communication/VariableStateMessageData";
import { VariableChangedToValueTrigger } from "./VariableTrigger";

jest.useFakeTimers();

describe('VariableTrigger test', () => {
  test('Triggered event has to be emitted after 100 ms. Condition variable >= 30', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.Or);
    const lhs = new VariableOperand(instanceApiMock, "owner", "variable");
    const condition = createCondition(lhs, ">=", 30);
    triggerCondition.addCondition(condition);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    jest.advanceTimersByTime(200);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
  });

  test('Triggered event has not to be emitted after 100 ms. Condition variable == 30, and value is 35', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.Or);
    const lhs = new VariableOperand(instanceApiMock, "owner", "variable");
    const condition = createCondition(lhs, "==", 30);
    triggerCondition.addCondition(condition);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    jest.advanceTimersByTime(200);

    expect(triggeredHandler).toHaveBeenCalledTimes(0);
  });

  test('Triggered event has not to be emitted after 100 ms, because the condition goes false.', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.Or);
    const lhs = new VariableOperand(instanceApiMock, "owner", "variable");
    const condition = createCondition(lhs, ">=", 30);
    triggerCondition.addCondition(condition);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 1000);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    setTimeout(() => {
      const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 0, 21, performance.now());
      const msg = new VariableChangedMessage('msgid', '', '', data);
      remoteVariable.onMessage(msg);
    }, 500);

    jest.advanceTimersByTime(1500);

    expect(triggeredHandler).toHaveBeenCalledTimes(0);
  });

  test('If the variable value changes twice, incase of greater than operator, only one event has to be emitted.', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.Or);
    const lhs = new VariableOperand(instanceApiMock, "owner", "variable");
    const condition = createCondition(lhs, ">=", 30);
    triggerCondition.addCondition(condition);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 500);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    setTimeout(() => {
      const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 40, 35, performance.now());
      const msg = new VariableChangedMessage('msgid', '', '', data);
      remoteVariable.onMessage(msg);
    }, 200);

    jest.advanceTimersByTime(1500);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
  });

  test('Triggered event has to be emitted, when the variable value is between 30 and 40.', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.And);
    const condition1 = createCondition(new VariableOperand(instanceApiMock, "owner", "variable"), ">", 30);
    const condition2 = createCondition(new VariableOperand(instanceApiMock, "owner", "variable"), "<", 40);
    triggerCondition.addCondition(condition1);
    triggerCondition.addCondition(condition2);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    jest.advanceTimersByTime(200);

    expect(triggeredHandler).toHaveBeenCalledTimes(1);
  });

  test('Triggered event has not to be emitted, first the variable value is between 30 and 40, but then value changes outside of the range.', async () => {
    const remoteVariable = new RemoteVariable("variable", "");

    const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };

    const triggerCondition = new ConditionGroup(OperatorType.And);
    const condition1 = createCondition(new VariableOperand(instanceApiMock, "owner", "variable"), ">", 30);
    const condition2 = createCondition(new VariableOperand(instanceApiMock, "owner", "variable"), "<", 40);
    triggerCondition.addCondition(condition1);
    triggerCondition.addCondition(condition2);

    const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 35, 21, performance.now());
    const msg = new VariableChangedMessage('msgid', '', '', data);
    remoteVariable.onMessage(msg);

    const variableTrigger = new VariableChangedToValueTrigger(remoteVariable, triggerCondition, 1000);

    const triggeredHandler = jest.fn();
    variableTrigger.onTriggered(triggeredHandler);

    setTimeout(() => {
      const data = new VariableStateMessageData("variable", '', TypeSignature.Number, 20, 40, performance.now());
      const msg = new VariableChangedMessage('msgid', '', '', data);
      remoteVariable.onMessage(msg);
    }, 500);

    jest.advanceTimersByTime(1500);

    expect(triggeredHandler).toHaveBeenCalledTimes(0);
  });
});
