import { generateUniqueId } from "./Communication/IpcMessage";
import { parseMessage, VariableMessage} from "./Communication/MessageParser";
import { VariableAddedMessage } from "./Communication/VariableAddedMessage";
import { VariableChangedMessage } from "./Communication/VariableChangedMessage";
import { VariableGetRemoteResponse } from "./Communication/VariableGetRemoteResponse";
import { VariableMessageType } from "./Communication/VariableMessageType";
import { VariableStateMessageData } from "./Communication/VariableStateMessageData";
import { LocalVariable } from "./LocalVariable";
import { TypeSignature, VariableType } from "./Variable";

export class LocalVariableManager {
  private variables: Record<string | symbol, LocalVariable> = {};

  readonly proxy: any;

  /**
   * @param localInstanceid The instance id of the owner instance, eg.: taurus.
   * @param comm
   */
  constructor(private readonly localInstanceid: string, private readonly comm: any) {
    this.proxy = new Proxy(this.variables, {
      get: (obj, prop) => {
        const variable: LocalVariable = obj[prop];

        if(typeof(variable) == 'undefined'){
          console.warn(`Variable (${prop.toString()}), not exists. Thus cannot be accessed through proxy object!`);
          return undefined;
        }

        return Reflect.get(obj, prop);
      },

      set: (_obj, prop, _value): boolean => {
        console.warn(`Variable (${prop.toString()}), can not be set through proxy. The form of ${prop.toString()}.value = <x> has to be used!`);
        return true;
      }
    });
  }

  createVariable(name: string, typeSignature: TypeSignature, ownerId: string, defaultValue: VariableType): LocalVariable {
    if (!this.variables[name]) {
      const variable = new LocalVariable(name, typeSignature, ownerId, defaultValue);

      variable.onChangedInternal((variable: LocalVariable) => {
        const messageData: VariableStateMessageData = VariableStateMessageData.create(variable);

        const messageId = generateUniqueId(variable.ownerId);
        const message: VariableChangedMessage = new VariableChangedMessage(messageId, variable.ownerId, 'placeholder', messageData);

        this.sendMessage(message);
      });

      this.addVariable(variable);
    }

    return this.variables[name];
  }

  getVariable(name: string): LocalVariable {
    return this.variables[name];
  }

  receiveMessage(jsonMessage: string): void {
    try {
      const message = parseMessage(jsonMessage);

      if (message !== null) {
        switch(message.messageType) {
          case VariableMessageType.GetRemote: {
            const variable: LocalVariable = this.variables[message.messageData.variableName];

            let messageData: VariableStateMessageData;
            let response: VariableGetRemoteResponse;

            if (variable != undefined) {
              messageData = VariableStateMessageData.create(variable);
              response = new VariableGetRemoteResponse(message.messageId, message.source, variable.ownerId, messageData);
            } else {
              messageData = VariableStateMessageData.createUndefined(message.messageData.variableName, this.localInstanceid);
              response = new VariableGetRemoteResponse(message.messageId, message.source, this.localInstanceid, messageData);
            }

            this.sendMessage(response);
            break;
          }
        }
      }
    } catch (error: any) {
      console.warn(error.message);
    }
  }

  private sendMessage(message: VariableMessage): void {
    this.comm.send(message);
  }

  private addVariable(variable: LocalVariable): void {
    if (!this.variables[variable.name]) {
      this.variables[variable.name] = variable;
      this.sendVariableAdded(this.variables[variable.name]);
    }
  }

  private sendVariableAdded(variable: LocalVariable) {
    const messageData: VariableStateMessageData = VariableStateMessageData.create(variable);

    const messageId = generateUniqueId(variable.ownerId);

    const message: VariableAddedMessage = new VariableAddedMessage(messageId, variable.ownerId, 'placeholder', messageData);

    this.sendMessage(message);
  }
}