import { VariableAddedMessage } from '../Communication/VariableAddedMessage';
import { VariableChangedMessage } from '../Communication/VariableChangedMessage';
import { VariableGetRemoteMessage } from '../Communication/VariableGetRemoteMessage';
import { VariableGetRemoteResponse } from '../Communication/VariableGetRemoteResponse';
import { VariableMessageType } from '../Communication/VariableMessageType';
import { VariableMesssageBroker } from '../VariableMessageBroker';

describe('VariableMesssageBroker', () => {
  let scriptrunner: any;
  let variableMesssageBroker: VariableMesssageBroker;

  test('should create VariableMessageBroker instance successfully', () => {
    expect(() => {
            new VariableMesssageBroker(scriptrunner);
          }).not.toThrow();
    expect(new VariableMesssageBroker(scriptrunner)).toBeInstanceOf(VariableMesssageBroker);
  });

  describe('receiveMessage', () => {
    beforeEach(() => {
      jest.clearAllMocks();
      scriptrunner = {
        sendMessageToScript: jest.fn(),
      };
      variableMesssageBroker = new VariableMesssageBroker(scriptrunner);
    });

    test('should run without error with a valid JSON string', () => {
      const jsonMessage = JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage);
      expect(() => {
        variableMesssageBroker.receiveMessage(jsonMessage);
      }).not.toThrow();
    });

    test('should console.warn error if there is a wrong JSON string', () => {
      const consoleWarnSpy = jest.spyOn(global.console, 'warn');
      const jsonMessage = 'NOT a JSON string';
      variableMesssageBroker.receiveMessage(jsonMessage);
      expect(consoleWarnSpy).toHaveBeenCalled();
      expect(consoleWarnSpy).toHaveBeenCalledTimes(1);
    });

    test('should add subscriber to a new subscription and send message to script with GetRemote messageType', () => {
      const jsonMessage = JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage);
      variableMesssageBroker.receiveMessage(jsonMessage);
      expect(scriptrunner.sendMessageToScript).toHaveBeenCalledWith('1', JSON.parse(jsonMessage));
      expect(variableMesssageBroker['subscriptionManager'].getSubscribers('1', 'cica')).toEqual(['subscriber1']);
    });

    test('should add new subscriber to subscription with GetRemote message type', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '2',
        source: 'subscriber2',
        messageData: {
          variableName: 'cica2'
        }
      } as VariableGetRemoteMessage));

      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '2',
        source: 'subscriber3',
        messageData: {
          variableName: 'cica2'
        }
      } as VariableGetRemoteMessage));
      expect(variableMesssageBroker['subscriptionManager'].getSubscribers('2', 'cica2')).toEqual(['subscriber2', 'subscriber3']);
    });

    test('should not add the same subscriber to the same subscription with GetRemote message type', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage));

      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage));
      expect(variableMesssageBroker['subscriptionManager'].getSubscribers('1', 'cica')).toEqual(['subscriber1']);
    });

    test('should return empty array if .getSubscribers method does not found a subscriber', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage));
      expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(1);
      expect(variableMesssageBroker['subscriptionManager'].getSubscribers('2', 'cica')).toEqual([]);
    });

    test('should send message to script with GetRemoteResponse messageType', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemoteResponse,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteResponse));
      expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(1);
    });

    test('should send message to all subscribers with Added messageType', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage));

      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.Added,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableAddedMessage));
      expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(2);
    });

    test('should send message to all subscribers with Changed messageType', () => {
      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.GetRemote,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableGetRemoteMessage));

      variableMesssageBroker.receiveMessage(JSON.stringify({
        messageType: VariableMessageType.Changed,
        instanceId: '1',
        source: 'subscriber1',
        messageData: {
          variableName: 'cica'
        }
      } as VariableChangedMessage));
      expect(scriptrunner.sendMessageToScript).toHaveBeenCalledTimes(2);
    });

  });
});
