import { VariableOperand } from "./VariableOperand";

describe('VariableOperand', () => {

  test('should create a VariableOperand successfully', () => {

    const instanceApiMock = {
      getInstanceById: jest.fn().mockReturnValue({
        variables: {
          test: {
            onChanged: jest.fn()
          }
        }
      })
    };

    expect(() => {
      new VariableOperand(instanceApiMock, 'test', 'test');
    }).not.toThrow();
  });

  test('should save variable value when operand is initialized', () => {
    const instanceApiMock = {
      getInstanceById: jest.fn().mockImplementation(() => {
        return {
          variables:
          {
            test:
            {
              value: 1,
              onChanged: jest.fn()
            }
          }
        };
      })
    };
    const variableOperand = new VariableOperand(instanceApiMock, 'test', 'test');

    expect(variableOperand.value).toBe(1);
  });

  test('should emit changed value, when change event happens', (done) => {
    jest.useFakeTimers();
    jest.spyOn(global, 'setTimeout');

    const instanceApiMock = {
      getInstanceById: jest.fn().mockReturnValue({
        variables: {
          test: {
            value: 1,
            onChanged: jest.fn().mockImplementation((callback) => {
              setTimeout(() => {
                callback(2, 1);
              }, 1000);
            })
          }
        }
      })
    };
    const variableOperand = new VariableOperand(instanceApiMock, 'test', 'test');
    const emitSpy = jest.spyOn(variableOperand, 'emit');

    expect(variableOperand.value).toBe(1);
    expect(emitSpy).not.toBeCalled();

    jest.advanceTimersByTime(2000);

    expect(emitSpy).toBeCalledWith("changed", 2);

    jest.useRealTimers();
    done();
  });

  test('should get variable value', () => {
    const instanceApiMock = {
      getInstanceById: jest.fn().mockImplementation(() => {
        return {
          variables: {
            test: {
              value: 1,
              onChanged: jest.fn()
            }
          }
        };
      })
    };
    const variableOperand = new VariableOperand(instanceApiMock, 'test', 'test');

    expect(variableOperand.value).toBe(1);

    const updatedValue = 2;
    jest.spyOn(VariableOperand.prototype, 'value', 'get').mockReturnValue(updatedValue);
    expect(variableOperand.value).toBe(updatedValue);
  });

  test('should get lastModifiedMs', () => {

    const instanceApiMock = {
      getInstanceById: jest.fn().mockImplementation(() => {
        return {
          variables: {
            test: {
              value: 1,
              lastModifiedMs: 1234567,
              onChanged: jest.fn()
            }
          }
        };
      })
    };
    const variableOperand = new VariableOperand(instanceApiMock, 'test', 'test');

    expect(variableOperand.lastModifiedMs).toBe(1234567);

    const updatedLastModifiedMs = 7654321;
    jest.spyOn(VariableOperand.prototype, 'lastModifiedMs', 'get').mockReturnValue(updatedLastModifiedMs);
    expect(variableOperand.lastModifiedMs).toBe(updatedLastModifiedMs);
  });

});
