"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModuleDescriptor = exports.ModuleDescriptorData = void 0;
const promises_1 = require("fs/promises");
const ActionsDescriptor_1 = require("./ActionsDescriptor");
const EventDescriptor_1 = require("./EventDescriptor");
const MethodDescriptor_1 = require("./MethodDescriptor");
const ModuleDescriptorObject_1 = require("./ModuleDescriptorObject");
const RuleDescriptor_1 = require("./RuleDescriptor");
const VariableDescriptor_1 = require("./VariableDescriptor");
const fs_1 = require("fs");
const strict_1 = __importDefault(require("node:assert/strict"));
const TriggerDescriptor_1 = require("./TriggerDescriptor");
class ModuleDescriptorData {
    constructor() {
        this.methods = [];
        this.events = [];
        this.rules = [];
        this.variables = [];
    }
}
exports.ModuleDescriptorData = ModuleDescriptorData;
class ModuleDescriptor extends ModuleDescriptorData {
    constructor() {
        super();
        this.methodsFromEntryCodeData = {};
    }
    readDescriptor(descriptorPath, entryCodePath) {
        const object = JSON.parse((0, fs_1.readFileSync)(descriptorPath, { encoding: 'utf8' }));
        if (entryCodePath) {
            const entryCode = (0, fs_1.readFileSync)(entryCodePath, { encoding: 'utf8' });
            return this.loadFromDescriptorObject(object, entryCode);
        }
        return this.loadFromDescriptorObject(object);
    }
    loadFromDescriptorObject(moduleDescriptorObject, moduleEntryCode) {
        if (moduleEntryCode) {
            this.methodsFromEntryCodeData = MethodDescriptor_1.MethodDescriptor.getMethodsFromEntryCode(moduleEntryCode);
        }
        for (const device in moduleDescriptorObject.Devices) {
            if (typeof moduleDescriptorObject.Devices[device].rules == 'undefined') {
                moduleDescriptorObject.Devices[device].rules = [];
            }
            if (typeof moduleDescriptorObject.Devices[device].extensions == 'undefined') {
                moduleDescriptorObject.Devices[device].extensions = {};
            }
            this.loadMethods(moduleDescriptorObject.Devices[device], moduleEntryCode);
            this.loadVariables(moduleDescriptorObject.Devices[device]);
            this.loadEvents(moduleDescriptorObject.Devices[device]);
            this.loadRules(moduleDescriptorObject.Devices[device]);
        }
    }
    loadFromModuleDescriptorData(moduleDescriptorData) {
        this.methods = moduleDescriptorData.methods;
        this.events = moduleDescriptorData.events;
        this.rules = moduleDescriptorData.rules;
        this.variables = moduleDescriptorData.variables;
    }
    loadMethods(device, moduleEntryCode) {
        const load = (data, factory) => {
            for (const [key, child] of Object.entries(data)) {
                if (child.type === ModuleDescriptorObject_1.ElementType.Method) {
                    this.methods.push(this.initMethod(key, child, factory, moduleEntryCode));
                }
            }
        };
        load(device.children, true);
        load(device.extensions, false);
    }
    initMethod(key, child, factory, moduleEntryCode) {
        var _a, _b;
        const actions = new ActionsDescriptor_1.ActionsDescriptor(moduleEntryCode ? this.methodsFromEntryCodeData[key].code : '', (_a = child.actions) === null || _a === void 0 ? void 0 : _a.type, (_b = child.actions) === null || _b === void 0 ? void 0 : _b.steps);
        return new MethodDescriptor_1.MethodDescriptor(key, child.parameters, actions, moduleEntryCode ? this.methodsFromEntryCodeData[key].description : child.description, moduleEntryCode ? this.methodsFromEntryCodeData[key].returnType : child.returnType, factory, child.orderId);
    }
    loadEvents(device) {
        const load = (data, factory) => {
            for (const [key, child] of Object.entries(data)) {
                if (child.type === ModuleDescriptorObject_1.ElementType.Event) {
                    this.events.push(this.initEvents(key, child, factory));
                }
            }
        };
        load(device.children, true);
        load(device.extensions, false);
    }
    initEvents(key, child, factory) {
        var _a, _b, _c;
        const actions = new ActionsDescriptor_1.ActionsDescriptor((_a = child.actions) === null || _a === void 0 ? void 0 : _a.code, (_b = child.actions) === null || _b === void 0 ? void 0 : _b.type, (_c = child.actions) === null || _c === void 0 ? void 0 : _c.steps);
        return new EventDescriptor_1.EventDescriptor(key, child.callbackParameters, actions, child.description, factory, child.orderId, child.baseEventTemplateName, child.parameterValues);
    }
    loadVariables(device) {
        const load = (data, factory) => {
            for (const [key, child] of Object.entries(data)) {
                if (child.type === ModuleDescriptorObject_1.ElementType.Variable) {
                    const variableDescriptor = new VariableDescriptor_1.VariableDescriptor(key, child.valueType, child.defaultValue, child.description, child.orderId, factory);
                    this.variables.push(variableDescriptor);
                }
            }
        };
        load(device.children, true);
        load(device.extensions, false);
    }
    loadRules(device) {
        const convertedRules = this.convertRules(device.rules);
        convertedRules.map((rule) => {
            var _a, _b, _c;
            const actions = new ActionsDescriptor_1.ActionsDescriptor((_a = rule.actions) === null || _a === void 0 ? void 0 : _a.code, (_b = rule.actions) === null || _b === void 0 ? void 0 : _b.type, (_c = rule.actions) === null || _c === void 0 ? void 0 : _c.steps);
            this.rules.push(new RuleDescriptor_1.RuleDescriptor(rule.name, rule.enabled, actions, rule.feId, rule.triggers, rule.condition, rule.factory, rule.orderId));
        });
    }
    /**
     * @brief Compares the interface of the modules. The User additions are ignored.
     *        Method signatures and event callback signatures are compared.
     *        The method and event codes are ignored.
     * @param other The other ModuleDescrirptor to compare with.
     * @returns true if the Moduledescriptors are compatible with each other.
     */
    isCompatibleWith(other) {
        for (const configMethod of this.methods) {
            if (!configMethod.factory)
                continue;
            const factoryMehtod = other.methods.find((factoryMethod) => factoryMethod.name === configMethod.name);
            if (factoryMehtod === undefined || !this.compareMethodSignature(factoryMehtod, configMethod)) {
                return false;
            }
        }
        for (const configEvent of this.events) {
            if (!configEvent.factory)
                continue;
            const factoryEvent = other.events.find((factoryEvent) => factoryEvent.name === configEvent.name);
            if (factoryEvent === undefined ||
                !this.compareParameters(factoryEvent.callbackParameters, configEvent.callbackParameters)) {
                return false;
            }
        }
        for (const configVariable of this.variables) {
            if (!configVariable.factory)
                continue;
            const factoryVariable = other.variables.find((factoryVariable) => factoryVariable.name === configVariable.name);
            let deepEqual = true;
            try {
                strict_1.default.deepStrictEqual(factoryVariable, configVariable);
            }
            catch (e) {
                deepEqual = false;
            }
            if (factoryVariable === undefined || !deepEqual) {
                return false;
            }
        }
        return true;
    }
    compareMethodSignature(originalMethod, newMethod) {
        if (newMethod.returnType != originalMethod.returnType)
            return false;
        if (!this.compareParameters(originalMethod.parameters, newMethod.parameters))
            return false;
        return true;
    }
    compareParameters(params1, params2) {
        if ((typeof params1 === 'undefined' && typeof params2 !== 'undefined') ||
            (typeof params1 !== 'undefined' && typeof params2 === 'undefined')) {
            return false;
        }
        else if (typeof params1 === 'undefined' && typeof params2 === 'undefined') {
            return true;
        }
        // Check parameter count
        if (params1.length != params2.length)
            return false;
        // Check parameter names, types
        for (let i = 0; i < params2.length; i++) {
            if (params2[i].name != params1[i].name)
                return false;
            if (params2[i].type != params1[i].type)
                return false;
        }
        return true;
    }
    /**
     * @brief   Gets the extension descriptor of the module, basically returns all the elements
     *          which factory property equals to false.
     * @returns ModuleDescriptorData with the user additive elments.
     */
    getExtensions() {
        const data = new ModuleDescriptorData();
        for (const method of this.methods) {
            if (method.factory == false) {
                data.methods.push(method);
            }
        }
        for (const event of this.events) {
            if (event.factory == false) {
                data.events.push(event);
            }
        }
        for (const variable of this.variables) {
            if (variable.factory == false) {
                data.variables.push(variable);
            }
        }
        data.rules = this.rules;
        return data;
    }
    /**
     * @brief   Check if the rules in the module are in the old format (event instead of trigger)
     * @returns void
     */
    convertRules(rules) {
        if (rules.length > 0) {
            rules.map((rule) => {
                if (rule.event && !rule.triggers) {
                    rule.triggers = [{
                            type: TriggerDescriptor_1.TriggerType.Event,
                            details: {
                                instanceId: rule.event.instanceId,
                                event: rule.event.event
                            }
                        }];
                    delete rule.event;
                }
                if (typeof rule.enabled == "undefined") {
                    rule.enabled = true;
                }
            });
        }
        return rules;
    }
    async saveDescriptor(path, moduleName, moduleId) {
        const [childrenMethods, extensionMethods] = this.seperateChildrenAndExtensions(this.methods);
        const [childrenVariables, extensionVariables] = this.seperateChildrenAndExtensions(this.variables);
        const [childrenEvents, extensionEvents] = this.seperateChildrenAndExtensions(this.events);
        const children = {};
        childrenMethods.map((descriptor) => {
            children[descriptor.name] = {
                ...descriptor,
                name: undefined,
                factory: undefined,
                type: 'method',
                actions: { type: descriptor.actions.type, steps: descriptor.actions.steps, code: '' }
            };
        });
        childrenVariables.map((descriptor) => {
            children[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'variable' };
        });
        childrenEvents.map((descriptor) => {
            children[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'event' };
        });
        const extensions = {};
        extensionMethods.map((descriptor) => {
            extensions[descriptor.name] = {
                ...descriptor,
                name: undefined,
                factory: undefined,
                type: 'method',
                actions: { type: descriptor.actions.type, steps: descriptor.actions.steps, code: '' }
            };
        });
        extensionVariables.map((descriptor) => {
            extensions[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'variable' };
        });
        extensionEvents.map((descriptor) => {
            extensions[descriptor.name] = { ...descriptor, name: undefined, factory: undefined, type: 'event' };
        });
        const moduleDescriptorObject = this.createModuleDescriptorObject({ children, extensions, moduleName, moduleId });
        await (0, promises_1.writeFile)(path, JSON.stringify(moduleDescriptorObject, undefined, 2));
    }
    seperateChildrenAndExtensions(descriptors) {
        const children = [];
        const extensions = [];
        descriptors.map((descriptor) => {
            if (descriptor.factory) {
                children.push(descriptor);
            }
            else {
                extensions.push(descriptor);
            }
        });
        return [children, extensions];
    }
    createModuleDescriptorObject({ children, extensions, moduleName, moduleId }) {
        const rules = this.rules.map((r) => ({ ...r, factory: undefined, type: undefined }));
        return {
            Schemas: {},
            Devices: {
                [moduleId]: {
                    name: moduleName,
                    type: 'group',
                    children,
                    extensions,
                    rules
                }
            }
        };
    }
}
exports.ModuleDescriptor = ModuleDescriptor;
//# sourceMappingURL=ModuleDescriptor.js.map