"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleManager = void 0;
const condition_api_1 = require("@lightware/condition-api");
const Rule_1 = require("./Rule");
const Triggers_1 = require("../Triggers/Triggers");
const Trigger_1 = require("../Triggers/Trigger");
const EventTrigger_1 = require("../Triggers/EventTrigger");
const VariableTrigger_1 = require("../Triggers/VariableTrigger");
const module_descriptor_1 = require("@lightware/module-descriptor");
const TimeTrigger_1 = __importDefault(require("../Triggers/TimeTrigger"));
const VariableChangedTrigger_1 = __importDefault(require("../Triggers/VariableChangedTrigger"));
class RuleManager {
    constructor(instanceApi) {
        this.instanceApi = instanceApi;
        this.rules = [];
    }
    init() {
        const moduleDescriptorJSON = this.instanceApi.getModuleData(this.instanceApi.instanceDir);
        const moduleDescriptor = new module_descriptor_1.ModuleDescriptor();
        moduleDescriptor.loadFromDescriptorObject(moduleDescriptorJSON);
        const events = this.getEvents(moduleDescriptor);
        this.rules = this.getRules(moduleDescriptor.rules, events);
    }
    setAction(ruleName, action) {
        const rule = this.rules.find((r) => ruleName === r.name);
        if (rule) {
            rule.setAction(action);
        }
        else {
            console.warn(`Rule with name ${ruleName} not found when trying to set an action for it in instance with id: ${this.instanceApi.instanceId}`);
        }
    }
    /**
     * The goal is to return the events that is from the current instance or a remote instance.
     * @param moduleDescriptor
     * @returns A map with all the instances and its corresponding events.
     */
    getEvents(moduleDescriptor) {
        let anotherInstanceIds = moduleDescriptor.rules.map((rule) => {
            return rule.triggers.map((trigger) => {
                if (trigger.type === Trigger_1.TriggerType.Event && trigger.details.instanceId !== "") {
                    return trigger.details.instanceId;
                }
                return "";
            });
        }).flat();
        anotherInstanceIds = anotherInstanceIds
            .filter((element, index) => (anotherInstanceIds.indexOf(element) === index) && element !== "");
        const events = { [this.instanceApi.instanceId]: moduleDescriptor.events };
        anotherInstanceIds.forEach((instanceId) => {
            const descriptorJSON = this.instanceApi.getModuleData(this.instanceApi.instanceDir.replace(this.instanceApi.instanceId, instanceId));
            const moduleDescriptor = new module_descriptor_1.ModuleDescriptor();
            moduleDescriptor.loadFromDescriptorObject(descriptorJSON);
            events[instanceId] = moduleDescriptor.events;
        });
        return events;
    }
    getRules(rules, events) {
        return rules.map((rule) => {
            const triggers = this.getTriggers(rule.triggers, events);
            const conditionGroup = rule.condition ? this.getConditionGroup(rule.condition, events) : undefined;
            return new Rule_1.Rule(rule.name, rule.enabled, triggers, conditionGroup);
        });
    }
    getTriggers(moduleDescriptorTriggers, events) {
        const triggers = new Triggers_1.Triggers();
        moduleDescriptorTriggers.map((trigger) => {
            switch (trigger.type) {
                case (Trigger_1.TriggerType.Event): {
                    const eventTrigger = new EventTrigger_1.EventTrigger(trigger.details.instanceId, trigger.details.event, this.instanceApi);
                    triggers.addTrigger(eventTrigger);
                    break;
                }
                case Trigger_1.TriggerType.Time: {
                    const options = {
                        cronTime: trigger.details.cron ? (0, module_descriptor_1.toCronString)(trigger.details.cron) : undefined,
                        startDate: trigger.details.startDate,
                        endDate: trigger.details.endDate
                    };
                    const timeTrigger = new TimeTrigger_1.default(options);
                    triggers.addTrigger(timeTrigger);
                    break;
                }
                case (Trigger_1.TriggerType.VariableChanged): {
                    const instance = this.instanceApi.getInstanceById(trigger.details.instanceId);
                    const remoteVariable = instance.variables[trigger.details.variable];
                    const variableTrigger = new VariableChangedTrigger_1.default(remoteVariable, trigger.duration);
                    triggers.addTrigger(variableTrigger);
                    break;
                }
                case (Trigger_1.TriggerType.VariableChangedFromTo): {
                    // TODO
                    break;
                }
                case Trigger_1.TriggerType.VariableChangedIntoRange:
                case (Trigger_1.TriggerType.VariableChangedToValue): {
                    const instance = this.instanceApi.getInstanceById(trigger.details.instanceId);
                    const remoteVariable = instance.variables[trigger.details.variable];
                    const conditionGroup = this.getConditionGroup(trigger.details.condition, events);
                    if (!conditionGroup) {
                        throw Error('VariableChangedToValueTrigger descriptor is invalid. Descriptor has to contain at least one condition!.');
                    }
                    let variableTrigger;
                    if (trigger.type == Trigger_1.TriggerType.VariableChangedToValue) {
                        variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, conditionGroup, trigger.duration);
                    }
                    else {
                        variableTrigger = new VariableTrigger_1.VariableChangedIntoRangeTrigger(remoteVariable, conditionGroup, trigger.duration);
                    }
                    triggers.addTrigger(variableTrigger);
                    break;
                }
            }
        });
        return triggers;
    }
    getConditionGroup(condition, events) {
        if (Object.keys(condition).length !== 0) {
            if (condition.type === 'condition') {
                const leftOperand = this.getOperand(condition.left, events);
                const rightOperand = this.getOperand(condition.right, events);
                return new condition_api_1.Condition(leftOperand, condition.operator, rightOperand, condition.duration);
            }
            else {
                const operatorType = condition.operator === module_descriptor_1.ConditionGroupOperator.AND ? condition_api_1.OperatorType.And : condition_api_1.OperatorType.Or;
                const conditionGroup = new condition_api_1.ConditionGroup(operatorType);
                condition.conditions.map((cond) => {
                    conditionGroup.addCondition(this.getConditionGroup(cond, events));
                });
                return conditionGroup;
            }
        }
        else {
            return undefined;
        }
    }
    getOperand(operandDescriptor, events) {
        switch (operandDescriptor.type) {
            case 'eventCallbackParameter': {
                const event = events[operandDescriptor.instanceId].find((e) => e.name === operandDescriptor.event);
                if (event) {
                    const index = event.getEventParameterIndex(operandDescriptor.name);
                    return new condition_api_1.EventParameterOperand(operandDescriptor.name, index);
                }
                throw new Error(`Could not find event with name ${operandDescriptor.event} on ${this.instanceApi.instanceId} instance`);
            }
            case 'StatusVariable': {
                return new condition_api_1.VariableOperand(this.instanceApi, operandDescriptor.instanceId, operandDescriptor.name);
            }
            case 'Constant': {
                return new condition_api_1.ConstOperand(operandDescriptor.value);
            }
            case 'InstanceParameter': {
                return new condition_api_1.ConstOperand(this.instanceApi.params[operandDescriptor.name]);
            }
        }
        throw new Error(`Operand type: ${operandDescriptor.type}, not supported`);
    }
}
exports.RuleManager = RuleManager;
//# sourceMappingURL=RuleManager.js.map