"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const module_descriptor_1 = require("@lightware/module-descriptor");
const RuleManager_1 = require("./RuleManager");
const condition_api_1 = require("@lightware/condition-api");
const EventTrigger_1 = require("../Triggers/EventTrigger");
const VariableTrigger_1 = require("../Triggers/VariableTrigger");
const Rule_1 = require("./Rule");
const TimeTrigger_1 = __importDefault(require("../Triggers/TimeTrigger"));
jest.mock('@lightware/condition-api');
jest.useFakeTimers();
const mockGetInstanceById = jest.fn(() => ({
    on: jest.fn(),
    variables: { testVar: { onChanged: jest.fn() } }
}));
const mockgetModuleData = jest.fn();
const mockInstanceApi = {
    getModuleData: mockgetModuleData,
    getInstanceById: mockGetInstanceById,
    instanceId: 'testInstance'
};
describe('RuleManager', () => {
    let ruleManager;
    test('Successfully create RuleManager with empty module descriptor', () => {
        mockgetModuleData.mockReturnValueOnce({
            Devices: {
                Test: {
                    name: 'test',
                    type: 'group',
                    children: {},
                    rules: [],
                    extensions: {}
                }
            }
        });
        expect(() => {
            ruleManager = new RuleManager_1.RuleManager(mockInstanceApi);
        }).not.toThrow();
    });
    describe('.getOperand', () => {
        test('Successfully return EventParameterOperand if operand type is eventCallbackParameter', () => {
            const operandDescriptor = {
                type: module_descriptor_1.OperandType.EventCallbackParameter,
                instanceId: 'testInstanceId',
                name: 'test',
                event: 'testEvent'
            };
            const events = {
                [operandDescriptor.instanceId]: [new module_descriptor_1.EventDescriptor('testEvent', [{
                            name: 'test',
                            title: 'test',
                            type: 'string'
                        }], new module_descriptor_1.ActionsDescriptor(), '')]
            };
            const operand = ruleManager['getOperand'](operandDescriptor, events);
            expect(operand instanceof condition_api_1.EventParameterOperand).toBe(true);
        });
        test('Throw an error if event with the name from the descriptor doesnt exist', () => {
            const operandDescriptor = {
                type: module_descriptor_1.OperandType.EventCallbackParameter,
                instanceId: 'testInstanceId',
                name: 'test',
                event: 'testEvent1'
            };
            const events = {
                [operandDescriptor.instanceId]: [new module_descriptor_1.EventDescriptor('testEvent', [{
                            name: 'test',
                            title: 'test',
                            type: 'string'
                        }], new module_descriptor_1.ActionsDescriptor(), '')]
            };
            expect(() => {
                ruleManager['getOperand'](operandDescriptor, events);
            }).toThrow('Could not find event with name testEvent1');
        });
        test('Successfully return VariableOperand if operand type is StatusVariable', () => {
            const operandDescriptor = {
                type: module_descriptor_1.OperandType.StatusVariable,
                instanceId: 'testInstanceId',
                name: 'test'
            };
            const operand = ruleManager['getOperand'](operandDescriptor, {});
            expect(operand instanceof condition_api_1.VariableOperand).toBe(true);
        });
        test('Successfully return ConstOperand if operand type is Constant', () => {
            const operandDescriptor = {
                type: module_descriptor_1.OperandType.Constant,
                value: 'test'
            };
            const operand = ruleManager['getOperand'](operandDescriptor, {});
            expect(operand instanceof condition_api_1.ConstOperand).toBe(true);
        });
        test('Throws an error if operand type is unsupported', () => {
            const operandDescriptor = { type: 'test' };
            expect(() => {
                ruleManager['getOperand'](operandDescriptor, {});
            }).toThrow('Operand type: test, not supported');
        });
    });
    describe('.getConditionGroup', () => {
        const conditionDescriptor = {
            type: 'condition',
            left: {
                type: module_descriptor_1.OperandType.StatusVariable,
                instanceId: 'testInstanceId',
                name: 'test'
            },
            operator: module_descriptor_1.Operator.Equals,
            right: {
                type: module_descriptor_1.OperandType.Constant,
                value: 'test'
            },
            duration: 0
        };
        test('Successfully return Condition instance if the type is condition', () => {
            const result = ruleManager['getConditionGroup'](conditionDescriptor, {});
            expect(result instanceof condition_api_1.Condition).toBe(true);
        });
        test('Successfully returns ConditionGroup instance if the type is conditionGroup with 2 conditions in it', () => {
            const conditionGroupDescriptor = {
                type: 'conditionGroup',
                operator: module_descriptor_1.ConditionGroupOperator.AND,
                conditions: [conditionDescriptor, conditionDescriptor]
            };
            const result = ruleManager['getConditionGroup'](conditionGroupDescriptor, {});
            expect(result instanceof condition_api_1.ConditionGroup).toBe(true);
        });
        test('Successfully returns ConditionGroup instance if the type is conditionGroup with 1 condition and 1 conditionGroup in it', () => {
            const innerConditionGroupDescriptor = {
                type: 'conditionGroup',
                operator: module_descriptor_1.ConditionGroupOperator.OR,
                conditions: [conditionDescriptor, conditionDescriptor]
            };
            const conditionGroupDescriptor = {
                type: 'conditionGroup',
                operator: module_descriptor_1.ConditionGroupOperator.AND,
                conditions: [innerConditionGroupDescriptor, conditionDescriptor]
            };
            const result = ruleManager['getConditionGroup'](conditionGroupDescriptor, {});
            expect(result instanceof condition_api_1.ConditionGroup).toBe(true);
        });
    });
    describe('.getTriggers', () => {
        describe('TimeTriggers', () => {
            test('Successfully returns TimeTrigger instance', () => {
                jest.setTimeout(1000);
                const triggersDescriptor = [
                    {
                        type: module_descriptor_1.TriggerType.Time,
                        duration: 0,
                        details: {
                            cron: {
                                seconds: '*',
                                minutes: '*',
                                hours: '*',
                                dayOfMonth: '*',
                                months: '*',
                                dayOfWeek: '*'
                            },
                            startDate: '2023-05-24 09:27:01',
                            endDate: '2024-05-24 09:27:01'
                        }
                    }
                ];
                const triggers = ruleManager['getTriggers'](triggersDescriptor, {});
                expect(triggers['triggers'].length).toBe(0);
                expect(triggers['triggers'][0] instanceof TimeTrigger_1.default).toBe(false);
            });
        });
        test('Successfully returns EventTrigger instance', () => {
            const triggersDescriptor = [
                {
                    type: module_descriptor_1.TriggerType.Event,
                    details: {
                        event: 'test',
                        instanceId: 'testInstanceId'
                    }
                }
            ];
            const triggers = ruleManager['getTriggers'](triggersDescriptor, {});
            expect(triggers['triggers'].length).toBe(1);
            expect(triggers['triggers'][0] instanceof EventTrigger_1.EventTrigger).toBe(true);
        });
        test('Successfully returns VariableChangedToValueTrigger instance', () => {
            jest.setTimeout(1000);
            const triggersDescriptor = [
                {
                    type: module_descriptor_1.TriggerType.VariableChangedToValue,
                    duration: 10000,
                    details: {
                        instanceId: 'testInstance',
                        variable: 'testVar',
                        condition: {
                            type: 'condition',
                            left: {
                                type: module_descriptor_1.OperandType.StatusVariable,
                                instanceId: 'testInstance',
                                name: 'testVar'
                            },
                            operator: module_descriptor_1.Operator.Equals,
                            right: {
                                type: module_descriptor_1.OperandType.Constant,
                                value: 50
                            },
                            duration: 0
                        }
                    }
                }
            ];
            const triggers = ruleManager['getTriggers'](triggersDescriptor, {});
            expect(triggers['triggers'].length).toBe(1);
            expect(triggers['triggers'][0] instanceof VariableTrigger_1.VariableChangedToValueTrigger).toBe(true);
        });
    });
    describe('.getRules', () => {
        test('Successfully returns a Rule with condition', () => {
            const ruleDescriptor = new module_descriptor_1.RuleDescriptor('ruleName', true, new module_descriptor_1.ActionsDescriptor(), '', [], {
                type: 'condition',
                left: {
                    type: module_descriptor_1.OperandType.StatusVariable,
                    instanceId: 'testInstance',
                    name: 'testVar'
                },
                operator: module_descriptor_1.Operator.Equals,
                right: {
                    type: module_descriptor_1.OperandType.Constant,
                    value: 50
                },
                duration: 0
            });
            const rules = ruleManager['getRules']([ruleDescriptor], {});
            ruleManager['rules'] = rules;
            expect(rules.length).toBe(1);
            expect(rules[0] instanceof Rule_1.Rule).toBe(true);
        });
        test('Successfully returns a Rule without condition', () => {
            const ruleDescriptor = new module_descriptor_1.RuleDescriptor('ruleName', true, new module_descriptor_1.ActionsDescriptor(), '', []);
            const rules = ruleManager['getRules']([ruleDescriptor], {});
            ruleManager['rules'] = rules;
            expect(rules.length).toBe(1);
            expect(rules[0] instanceof Rule_1.Rule).toBe(true);
        });
    });
    describe('.setAction', () => {
        test('Successfully set action for an already created rule', () => {
            expect(() => {
                ruleManager.setAction('ruleName', async () => {
                    console.log('test');
                });
            }).not.toThrow();
        });
        test('If rule can not be found call console.warn', () => {
            const warnSpy = jest.spyOn(console, 'warn').mockImplementation();
            ruleManager.setAction('test', async () => {
                console.log('test');
            });
            expect(warnSpy)
                .toHaveBeenCalledWith('Rule with name test not found when trying to set an action for it in instance with id: testInstance');
        });
    });
});
//# sourceMappingURL=RuleManager.test.js.map