"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const condition_api_1 = require("@lightware/condition-api");
const variable_api_1 = require("@lightware/variable-api");
const VariableChangedMessage_1 = require("@lightware/variable-api/dist/src/Communication/VariableChangedMessage");
const VariableStateMessageData_1 = require("@lightware/variable-api/dist/src/Communication/VariableStateMessageData");
const VariableTrigger_1 = require("./VariableTrigger");
jest.useFakeTimers();
describe('VariableTrigger test', () => {
    test('Triggered event has to be emitted after 100 ms. Condition variable >= 30', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.Or);
        const lhs = new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable");
        const condition = (0, condition_api_1.createCondition)(lhs, ">=", 30);
        triggerCondition.addCondition(condition);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        jest.advanceTimersByTime(200);
        expect(triggeredHandler).toHaveBeenCalledTimes(1);
    });
    test('Triggered event has not to be emitted after 100 ms. Condition variable == 30, and value is 35', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.Or);
        const lhs = new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable");
        const condition = (0, condition_api_1.createCondition)(lhs, "==", 30);
        triggerCondition.addCondition(condition);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        jest.advanceTimersByTime(200);
        expect(triggeredHandler).toHaveBeenCalledTimes(0);
    });
    test('Triggered event has not to be emitted after 100 ms, because the condition goes false.', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.Or);
        const lhs = new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable");
        const condition = (0, condition_api_1.createCondition)(lhs, ">=", 30);
        triggerCondition.addCondition(condition);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 1000);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        setTimeout(() => {
            const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 0, 21, performance.now());
            const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
            remoteVariable.onMessage(msg);
        }, 500);
        jest.advanceTimersByTime(1500);
        expect(triggeredHandler).toHaveBeenCalledTimes(0);
    });
    test('If the variable value changes twice, incase of greater than operator, only one event has to be emitted.', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.Or);
        const lhs = new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable");
        const condition = (0, condition_api_1.createCondition)(lhs, ">=", 30);
        triggerCondition.addCondition(condition);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 500);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        setTimeout(() => {
            const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 40, 35, performance.now());
            const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
            remoteVariable.onMessage(msg);
        }, 200);
        jest.advanceTimersByTime(1500);
        expect(triggeredHandler).toHaveBeenCalledTimes(1);
    });
    test('Triggered event has to be emitted, when the variable value is between 30 and 40.', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.And);
        const condition1 = (0, condition_api_1.createCondition)(new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable"), ">", 30);
        const condition2 = (0, condition_api_1.createCondition)(new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable"), "<", 40);
        triggerCondition.addCondition(condition1);
        triggerCondition.addCondition(condition2);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 100);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        jest.advanceTimersByTime(200);
        expect(triggeredHandler).toHaveBeenCalledTimes(1);
    });
    test('Triggered event has not to be emitted, first the variable value is between 30 and 40, but then value changes outside of the range.', async () => {
        const remoteVariable = new variable_api_1.RemoteVariable("variable", "");
        const instanceApiMock = { getInstanceById: jest.fn().mockReturnValue({ variables: { variable: remoteVariable } }) };
        const triggerCondition = new condition_api_1.ConditionGroup(condition_api_1.OperatorType.And);
        const condition1 = (0, condition_api_1.createCondition)(new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable"), ">", 30);
        const condition2 = (0, condition_api_1.createCondition)(new condition_api_1.VariableOperand(instanceApiMock, "owner", "variable"), "<", 40);
        triggerCondition.addCondition(condition1);
        triggerCondition.addCondition(condition2);
        const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 35, 21, performance.now());
        const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
        remoteVariable.onMessage(msg);
        const variableTrigger = new VariableTrigger_1.VariableChangedToValueTrigger(remoteVariable, triggerCondition, 1000);
        const triggeredHandler = jest.fn();
        variableTrigger.onTriggered(triggeredHandler);
        setTimeout(() => {
            const data = new VariableStateMessageData_1.VariableStateMessageData("variable", '', variable_api_1.TypeSignature.Number, 20, 40, performance.now());
            const msg = new VariableChangedMessage_1.VariableChangedMessage('msgid', '', '', data);
            remoteVariable.onMessage(msg);
        }, 500);
        jest.advanceTimersByTime(1500);
        expect(triggeredHandler).toHaveBeenCalledTimes(0);
    });
});
//# sourceMappingURL=VariableTrigger.test.js.map