"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoteVariableManager = void 0;
const IpcMessage_1 = require("./Communication/IpcMessage");
const MessageParser_1 = require("./Communication/MessageParser");
const VariableGetRemoteMessage_1 = require("./Communication/VariableGetRemoteMessage");
const RemoteVariable_1 = require("./RemoteVariable");
class RemoteVariableManager {
    constructor(ownerInstanceId, localInstanceid, comm) {
        this.ownerInstanceId = ownerInstanceId;
        this.localInstanceid = localInstanceid;
        this.comm = comm;
        this.variables = {};
        this.proxy = new Proxy(this.variables, {
            get: (obj, prop) => {
                if (typeof obj[prop] == "undefined") {
                    const warnMessage1 = `LARA variable ${prop.toString()} not exists on ${this.ownerInstanceId} or not ready yet.`;
                    const warnMessage2 = `If ${this.ownerInstanceId} does have LARA variable ${prop.toString()} then you probably need to wait for an appropriate event before accessing it.`;
                    console.warn(`${warnMessage1} ${warnMessage2}`);
                }
                if (obj[prop] === null || typeof obj[prop] == "undefined") {
                    obj[prop] = this.getVariable(prop.toString());
                }
                return Reflect.get(obj, prop);
            },
            set: (obj, prop, value) => {
                console.warn(`Variable ${prop.toString()}, owned by ${this.ownerInstanceId}, can not be set through proxy. The form of ${prop.toString()}.value = <x> has to be used!`);
                return true;
            }
        });
    }
    createVariablePlaceholder(variableName) {
        if (this.variables[variableName] === undefined) {
            this.variables[variableName] = null;
        }
    }
    getVariable(variableName) {
        if (this.variables[variableName] === undefined || this.variables[variableName] === null) {
            this.variables[variableName] = new RemoteVariable_1.RemoteVariable(variableName, this.ownerInstanceId);
            const messageData = new VariableGetRemoteMessage_1.VariableGetRemoteMessageData(variableName);
            const messageId = (0, IpcMessage_1.generateUniqueId)(this.ownerInstanceId);
            const message = new VariableGetRemoteMessage_1.VariableGetRemoteMessage(messageId, this.ownerInstanceId, 'placeholder', messageData);
            this.comm.send(message);
        }
        return this.variables[variableName];
    }
    receiveMessage(jsonMessage) {
        try {
            let message = (0, MessageParser_1.parseMessage)(jsonMessage);
            if (message !== null) {
                message = message;
                const remoteVariable = this.variables[message.messageData.variableName];
                if (remoteVariable !== null && remoteVariable !== undefined && remoteVariable.ownerId === message.messageData.ownerId) {
                    remoteVariable.onMessage(message);
                }
            }
        }
        catch (error) {
            console.warn(error.message);
        }
    }
}
exports.RemoteVariableManager = RemoteVariableManager;
//# sourceMappingURL=RemoteVariableManager.js.map