import { generateUniqueId } from "./Communication/IpcMessage";
import { parseMessage, VariableMessageOnRemote, VariableMessage } from "./Communication/MessageParser";
import { VariableGetRemoteMessage, VariableGetRemoteMessageData } from "./Communication/VariableGetRemoteMessage";
import { RemoteVariable } from "./RemoteVariable";

export class RemoteVariableManager {
  private variables: Record<string | symbol, RemoteVariable | null> = {};

  readonly proxy: any;

  constructor(private readonly ownerInstanceId: string, private readonly localInstanceid: string, private readonly comm: any) {
    this.proxy = new Proxy(this.variables, {
      get: (obj, prop) => {
        if (typeof obj[prop] == "undefined") {
          const warnMessage1 = `LARA variable ${prop.toString()} not exists on ${this.ownerInstanceId} or not ready yet.`;
          const warnMessage2 = `If ${this.ownerInstanceId} does have LARA variable ${prop.toString()} then you probably need to wait for an appropriate event before accessing it.`;
          console.warn(`${warnMessage1} ${warnMessage2}`);
        }

        if (obj[prop] === null || typeof obj[prop] == "undefined") {
          obj[prop] = this.getVariable(prop.toString())
        }

        return Reflect.get(obj, prop);
      },

      set: (obj, prop, value): boolean => {
        console.warn(`Variable ${prop.toString()}, owned by ${this.ownerInstanceId}, can not be set through proxy. The form of ${prop.toString()}.value = <x> has to be used!`);
        return true;
      }
    });
  }

  createVariablePlaceholder(variableName: string): void {
    if (this.variables[variableName] === undefined) {
      this.variables[variableName] = null;
    }
  }

  getVariable(variableName: string): RemoteVariable {
    if (this.variables[variableName] === undefined || this.variables[variableName] === null) {
      this.variables[variableName] = new RemoteVariable(variableName, this.ownerInstanceId);

      const messageData: VariableGetRemoteMessageData = new VariableGetRemoteMessageData(variableName);

      const messageId = generateUniqueId(this.ownerInstanceId);

      const message: VariableGetRemoteMessage = new VariableGetRemoteMessage(messageId, this.ownerInstanceId, 'placeholder', messageData);

      this.comm.send(message);
    }

    return this.variables[variableName]!;
  }

  receiveMessage(jsonMessage: string): void {
    try {
      let message: VariableMessage = parseMessage(jsonMessage);

      if (message !== null) {
        message = message as VariableMessageOnRemote;
        const remoteVariable: RemoteVariable | null | undefined = this.variables[message.messageData.variableName];

        if (remoteVariable !== null && remoteVariable !== undefined && remoteVariable.ownerId === message.messageData.ownerId) {
          remoteVariable.onMessage(message);
        }
      }
    } catch (error: any) {
      console.warn(error.message);
    }
  }
}
