const common = require('./common');
const variableApi = require('@lightware/variable-api');

/**
 * Instanciated by InstanceApi.getInstance method
 * Returns an interface to the local LARA module instance
 */
class InstanceHandler {
  methods = {};
  subscriptions = [];
  subscriptionCallbacks = {};
  frontendEventSubscriptions = [];
  userpanelChanges = {};
  variables = {};

  constructor(props) {
    this.instanceId = props.instanceId;
    this.instanceDir = props.instanceDir;
    this.comm = props.comm;

    this.moduleDescriptor = props.moduleDescriptor;
    this.packageData = props.packageData;

    this.LocalVariableManager = new variableApi.LocalVariableManager(this.instanceId, this.comm);
    this.variables = this.LocalVariableManager.proxy;

    // We can do it with IPC communication
    // Or through worker thread's postMessage function
    this.comm.on('message', async data => {
      this.LocalVariableManager.receiveMessage(data)
      data = JSON.parse(data);

      switch (data.type) {
        case 'callMethod':
          if (typeof this.methods[data.methodName] != 'undefined') {
            try {
              let result = await this.methods[data.methodName](...data.params);

              this.comm.send({
                messageId: data.messageId,
                type: 'result',
                resultType: 'resolve',
                result: result
              });
            } catch (e) {
              console.error(e);

              this.comm.send({
                messageId: data.messageId,
                type: 'result',
                resultType: 'reject',
                result: e.message
              });
            }
          }
          break;
        case 'subscribeToEvent':
          if (typeof this.subscriptionCallbacks[data.event] !== 'undefined') {
            await this.subscriptionCallbacks[data.event].subscribeCallback();
          }
          break;
        case 'unsubscribeFromEvent':
          if (typeof this.subscriptionCallbacks[data.event] !== 'undefined') {
            await this.subscriptionCallbacks[data.event].unsubscribeCallback();
          }
          break;
        case 'getInitData':
          this.sendMessageToUserPanel({
            type: 'initData',
            instanceId: this.instanceId,
            params: {
              eventSubscriptions: JSON.parse(JSON.stringify(this.frontendEventSubscriptions)),
              userpanelChanges: JSON.parse(JSON.stringify(this.userpanelChanges))
            }
          });

          if (data.variables && data.variables.length > 0) {
            this.comm.send({
              messageId: 1,
              type: 'initVariables',
              instanceId: this.instanceId,
              variables: data.variables
            });
          }
          break;
        case 'uiChange':
          if (!data.params) {
            data.params = {};
          }

          if (!data.params.args) {
            data.params.args = [];
          }

          this.dispatchEvent(data.event, ...data.params.args);
          break;
        case 'dispatchEvent':
          if (!data.args) {
            data.args = [];
          }
          // Dispatch event locally
          this.dispatchEventInternal(data.event, ...data.args);
          break;
      }
    });

    this.generateInterface();
  }

  createVirtualMethods(items) {
    for (const [key, value] of Object.entries(items)) {
      if (value.type == 'method') {
        if (typeof this[key] == 'undefined') {
          this[key] = (...args) => {
            return new Promise(async (resolve, reject) => {
              const res = common.checkArguments(value.parameters, args);

              if (res.success) {
                let result = await this.methods[key](...res.args);
                resolve(result);
              } else {
                reject(new Error(res.errorMsg));
              }
            });
          };
        }
      }
    }
  }

  createVariables(items) {
    for (const [key, value] of Object.entries(items)) {
      if (value.type == 'variable') {
        this.LocalVariableManager.createVariable(key, items[key].valueType, this.instanceId, items[key].defaultValue);
        this.variables[key].onChanged((current, previous) => {
          common.updateVariable(key, current, this.instanceId, this.comm);
        });
      }
    }
  }

  generateInterface() {
    if (this.moduleDescriptor.Devices) {
      for (const [devicekey, device] of Object.entries(this.moduleDescriptor.Devices)) {
        if (device.children) {
          this.createVirtualMethods(device.children);
          this.createVariables(device.children);
        }

        if (device.extensions) {
          this.createVirtualMethods(device.extensions);
          this.createVariables(device.extensions);
        }
      }
    }
  }


  getLocalStoragePath() {
    return `${this.instanceDir}/localstorage`;
  }

  registerMethod(methodName, callback) {
    this.methods[methodName] = callback;
  }

  async on(event, callback) {
    if (typeof this.subscriptions[event] == 'undefined') {
      this.subscriptions[event] = [];
    }

    this.subscriptions[event].push(callback);

    if (typeof this.subscriptionCallbacks[event] !== 'undefined') {
      await this.subscriptionCallbacks[event].subscribeCallback();
    }
  }

  dispatchEventInternal(event, ...args) {
    // Handle local subscribers
    if (typeof this.subscriptions[event] != 'undefined') {
      if (this.subscriptions[event].length > 0) {
        try {
          this.subscriptions[event].map(callback => {
            if (typeof callback == 'function') {
              callback(...(args || []));
            }
          });
        } catch (e) {
          console.error(`Error: event subscription storage contains invalid elements!`);
        }
      }
    }
  }

  dispatchEvent(event, ...args) {
    // Handle local subscribers
    this.dispatchEventInternal(event, ...args);

    // Dispatch event to remote subscribers
    this.comm.send({
      messageId: 1,
      type: 'dispatchEvent',
      event: event,
      args: args
    });
  }

  registerEvent(eventName, subscribeCallback, unsubscribeCallback) {
    // If there were subscriptions before the event subscription callback has been registered, invoke the subscription callback
    if (this.subscriptions[eventName] && this.subscriptions[eventName].length > 0) {
      for (let i = 0; i < this.subscriptions[eventName].length; i++) {
        subscribeCallback();
      }
    }
    this.subscriptionCallbacks[eventName] = { subscribeCallback, unsubscribeCallback };
  }

  registerProperty(propertyName, propertyReference) {
    this.properties[propertyName] = propertyReference;
  }

  addFrontendEventSubscription(eventName, elementId, event) {
    let subscriptionIndex = this.frontendEventSubscriptions.findIndex(item => {
      return item.eventName == eventName && item.elementId == elementId && item.event == event;
    });

    if (subscriptionIndex == -1) {
      this.frontendEventSubscriptions.push({
        eventName: eventName,
        elementId: elementId,
        event: event
      });
    }
  }

  sendMessageToUserPanel(message) {
    this.comm.send({
      type: 'sendMessageToUserPanel',
      instanceId: this.instanceId,
      args: message
    });
  }

  updateStatus(label, value, style) {
    this.comm.send({
      type: 'updateStatus',
      instanceId: this.instanceId,
      args: {
        label,
        value,
        style
      }
    });
  }

  // UserPanel state change
  changeUserPanelContent(data) {
    if (typeof this.userpanelChanges[data.params.elementId] == 'undefined') {
      this.userpanelChanges[data.params.elementId] = {};
    }

    switch (data.type) {
      case 'HtmlInner':
        this.userpanelChanges[data.params.elementId]['HtmlInner'] = {
          innerHtml: data.params.innerHtml,
          isText: data.params.isText
        };
        break;
      case 'HtmlStyle':
        if (typeof this.userpanelChanges[data.params.elementId]['HtmlStyle'] == 'undefined') {
          this.userpanelChanges[data.params.elementId]['HtmlStyle'] = {};
        }
        this.userpanelChanges[data.params.elementId]['HtmlStyle'][data.params.cssPropertyName] =
          data.params.cssPropertyValue;
        break;
      case 'HtmlClass':
        if (typeof this.userpanelChanges[data.params.elementId]['HtmlClass'] == 'undefined') {
          this.userpanelChanges[data.params.elementId]['HtmlClass'] = {};
        }
        this.userpanelChanges[data.params.elementId]['HtmlClass'][data.params.cssClassName] =
          data.params.cssClassName;
        break;
      case 'HtmlElementValue':
        this.userpanelChanges[data.params.elementId]['HtmlElementValue'] = data.params.value;
        break;
    }

    this.sendMessageToUserPanel(data);
  }

  text(elementId, innerHtml) {
    this.changeUserPanelContent({
      type: 'HtmlInner',
      params: { elementId, innerHtml, isText: true }
    });
  }

  html(elementId, innerHtml) {
    this.changeUserPanelContent({
      type: 'HtmlInner',
      params: { elementId, innerHtml, isText: false }
    });
  }

  setValue(elementId, value) {
    this.changeUserPanelContent({
      type: 'HtmlElementValue',
      params: { elementId, value }
    });
  }

  css(elementId, cssPropertyName, cssPropertyValue) {
    this.changeUserPanelContent({
      type: 'HtmlStyle',
      params: { elementId, cssPropertyName, cssPropertyValue }
    });
  }

  hide(elementId) {
    this.changeUserPanelContent({
      type: 'HtmlStyle',
      params: { elementId, cssPropertyName: 'visibility', cssPropertyValue: 'hidden' }
    });
  }

  show(elementId) {
    this.changeUserPanelContent({
      type: 'HtmlStyle',
      params: { elementId, cssPropertyName: 'visibility', cssPropertyValue: 'visible' }
    });
  }

  addClass(elementId,cssClassName) {
    this.changeUserPanelContent({
      type: 'HtmlClass',
      params: { elementId, cssClassName, isClassRemove: false }
    });
  }

  removeClass(elementId,cssClassName) {
    this.changeUserPanelContent({
      type: 'HtmlClass',
      params: { elementId, cssClassName, isClassRemove: true }
    });
  }
}

exports.InstanceHandler = InstanceHandler;
